package ru.yandex.direct.core.entity.campaign.service.accesschecker.api5;

import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheckApiImpl;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignSourceUtils;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.RequestSource;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AbstractCampaignAccessibiltyChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignForAccessCheckRepositoryAdapter;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Имплементация CampaignAccessibiltyChecker, проверяющая кампанию на возможность работы с ней в апи
 * Пока проверяется только тип кампании
 */
public class Api5CampaignAccessibilityChecker extends AbstractCampaignAccessibiltyChecker<CampaignForAccessCheckApiImpl> {
    private static final Map<RequestSource, Api5CampaignAccessibilityChecker> API5_ACCESSIBILITY_CHECKER_BY_ACCESS_GROUP =
            StreamEx.of(RequestSource.values())
            .mapToEntry(requestSource -> new Api5CampaignAccessibilityChecker(
                    CampaignTypeKinds.API5_EDIT, CampaignTypeKinds.API5_VISIBLE,
                    CampaignSourceUtils.getApi5EditableSources(requestSource),
                    CampaignSourceUtils.getApi5VisibleSources(requestSource))
            ).toMap();

    private Api5CampaignAccessibilityChecker(
            Set<CampaignType> supportedTypes, Set<CampaignType> allowableTypes,
            Set<CampaignSource> editableSources, Set<CampaignSource> supportedSources) {
        super(supportedTypes, allowableTypes, editableSources, supportedSources);
    }

    public static Api5CampaignAccessibilityChecker getApi5AccessibilityChecker() {
        return getApi5AccessibilityChecker(RequestSource.API_DEFAULT);
    }

    public static Api5CampaignAccessibilityChecker getApi5AccessibilityChecker(RequestSource requestSource) {
        return requestSource == null ?
                API5_ACCESSIBILITY_CHECKER_BY_ACCESS_GROUP.get(RequestSource.API_DEFAULT) :
                API5_ACCESSIBILITY_CHECKER_BY_ACCESS_GROUP.get(requestSource);
    }

    /**
     * Дофильтровываем универсальные кампании. Они размечены через source, но для пользователя это отдельный тип,
     * поэтому делаем это в проверке типов, а не сорсов
     */
    @Override
    protected boolean isEditableType(CampaignForAccessCheckApiImpl campaign) {
        return super.isEditableType(campaign)
                && !nvl(campaign.getIsUniversal(), false);
    }

    /**
     * Дофильтровываем универсальные кампании. Они размечены через source, но для пользователя это отдельный тип,
     * поэтому делаем это в проверке типов, а не сорсов
     */
    @Override
    protected boolean isAllowableType(CampaignForAccessCheckApiImpl campaign) {
        //Универсальные кампании фильтруем тут, потому что для пользователя это отдельный тип
        return super.isAllowableType(campaign)
                && !nvl(campaign.getIsUniversal(), false);
    }

    @Override
    public CampaignForAccessCheckRepositoryAdapter<CampaignForAccessCheckApiImpl> toAllowableCampaignsRepositoryAdapter(ClientId clientId) {
        return new AllowableCampaignsApi5AccessCheckRepositoryAdapter(clientId, allowableTypes, allowableSources);
    }

    @Override
    public CampaignForAccessCheckRepositoryAdapter<CampaignForAccessCheckApiImpl> toAllCampaignsRepositoryAdapter(ClientId clientId) {
        return new AllClientCampaignsApi5AccessCheckRepositoryAdapter(clientId);
    }
}
