package ru.yandex.direct.core.entity.campaign.service.pricerecalculation;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.StatusAutobudgetShow;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.autobudget.model.AutobudgetForecast;
import ru.yandex.direct.core.entity.autobudget.model.AutobudgetForecastStatus;
import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetForecastRepository;
import ru.yandex.direct.core.entity.bids.container.SetPhrasesInitialPricesOption;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.entity.campaign.container.CampaignStrategyChangingSettings;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDefaultPriceRecalculation;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.TextCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils;
import ru.yandex.direct.core.entity.campaign.service.MailTextCreatorService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.mailnotification.model.EventStrategyParams;
import ru.yandex.direct.core.entity.mailnotification.model.StrategyEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.core.entity.statistics.container.ProcessedAuctionStat;
import ru.yandex.direct.core.entity.statistics.repository.BsAuctionStatRepository;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.MoneyUtils;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static ru.yandex.direct.utils.CommonUtils.max;
import static ru.yandex.direct.utils.CommonUtils.min;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для пересчета ставок при смене стратегий
 */
@Service
public class CommonCampaignPriceRecalculationService {
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AutobudgetForecastRepository autobudgetForecastRepository;
    private final AdGroupRepository adGroupRepository;
    private final RetargetingRepository retargetingRepository;
    private final CampaignService campaignService;
    private final BidService bidService;
    private final MailNotificationEventService mailNotificationEventService;
    private final MailTextCreatorService mailTextCreatorService;
    private final LogPriceService logPriceService;
    private final BidRepository bidRepository;
    private final BsAuctionStatRepository bsAuctionStatRepository;
    private final KeywordRepository keywordRepository;

    private static final int SCALE_FOR_MONEY = 4;
    private static final RoundingMode ROUNDING_MODE = RoundingMode.HALF_UP;
    private static final boolean DEFAULT_CALCULATING_BS_STATISTICS_STATUS = true;
    private static final Map<CampaignType, Boolean> CALCULATING_BS_STATISTICS_STATUS_BY_CAMPAIGN_TYPE = Map.of(
            CampaignType.TEXT, true,
            CampaignType.CONTENT_PROMOTION, false
    );

    public CommonCampaignPriceRecalculationService(ShardHelper shardHelper,
                                                   CampaignRepository campaignRepository,
                                                   AutobudgetForecastRepository autobudgetForecastRepository,
                                                   AdGroupRepository adGroupRepository,
                                                   RetargetingRepository retargetingRepository,
                                                   CampaignService campaignService,
                                                   BidService bidService,
                                                   MailNotificationEventService mailNotificationEventService,
                                                   MailTextCreatorService mailTextCreatorService,
                                                   LogPriceService logPriceService,
                                                   BidRepository bidRepository,
                                                   BsAuctionStatRepository bsAuctionStatRepository,
                                                   KeywordRepository keywordRepository
    ) {

        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.autobudgetForecastRepository = autobudgetForecastRepository;
        this.adGroupRepository = adGroupRepository;
        this.retargetingRepository = retargetingRepository;
        this.campaignService = campaignService;
        this.bidService = bidService;
        this.logPriceService = logPriceService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.mailTextCreatorService = mailTextCreatorService;
        this.bidRepository = bidRepository;
        this.bsAuctionStatRepository = bsAuctionStatRepository;
        this.keywordRepository = keywordRepository;
    }

    public AppliedChanges<Retargeting> applyRetargetingPriceChanges(Retargeting retargeting, BigDecimal newPrice) {
        return new ModelChanges<>(retargeting.getId(), Retargeting.class)
                .process(newPrice, Retargeting.PRICE_CONTEXT)
                .process(StatusBsSynced.NO, Retargeting.STATUS_BS_SYNCED)
                .applyTo(retargeting);
    }

    public <C extends CampaignWithStrategy> LogPriceData calculateLogPriceData(
            C campaign, BigDecimal newPriceContext, Long adGroupId, Long id, BigDecimal newPrice,
            LogPriceData.OperationType operationType) {
        return new LogPriceData(
                campaign.getId(),
                adGroupId,
                id,
                newPriceContext.doubleValue(),
                newPrice.doubleValue(),
                campaign.getCurrency(),
                operationType
        );
    }

    /**
     * Действия при изменения стратегии на кампаниях. Массовые действия
     */
    public <C extends CampaignWithDefaultPriceRecalculation> void afterTextCampaignsStrategyChanged(
            List<AppliedChanges<C>> appliedChanges,
            CampaignStrategyChangingSettings settings,
            Long operatorUid, UidAndClientId uidAndClientId) {
        textCampaignPriceChange(appliedChanges, settings, operatorUid, uidAndClientId);
        markStrategyChange(appliedChanges, operatorUid);
        mailNotification(appliedChanges, operatorUid, uidAndClientId);  //TODO перенести в AdditionalActionsContainer
    }

    /**
     * Пересчет ставок при смене стратегий.
     * Аналог функции campaign_strategy_changed в perl.
     * todo ssdmitriev https://st.yandex-team.ru/DIRECT-110954 рефакторинг
     */
    private <C extends CampaignWithDefaultPriceRecalculation> void textCampaignPriceChange(
            List<AppliedChanges<C>> appliedChanges,
            CampaignStrategyChangingSettings settings,
            long operator, UidAndClientId client) {
        restoreManualBids(
                filterList(appliedChanges, CampaignStrategyUtils::filterAutoBudgetToManual), settings, client);
        cleanAutobudgetForecast(filterList(appliedChanges, CampaignStrategyUtils::filterAutoBudgetToManual), client);
        enableAutoBudget(filterList(appliedChanges, CampaignStrategyUtils::filterEnableAutoBudget), operator, client);
        setPhrasesPrices(filterList(appliedChanges, CampaignStrategyUtils::filterToManual), operator, client);
        setBidsRetargeting(appliedChanges, settings, operator, client);

        List<Long> campaignIdsWithAutobudget =
                filterAndMapList(appliedChanges, CampaignStrategyUtils::filterToAutobudget, c -> c.getModel().getId());
        resetBidsRetargetingsBsSyncStatus(shardHelper.getShardByClientId(client.getClientId()),
                campaignIdsWithAutobudget);
    }

    /**
     * Установка начальных цен на поиске и в сети и копирование цены с сети на поиск
     */
    private <C extends CampaignWithDefaultPriceRecalculation> void setPhrasesPrices(List<AppliedChanges<C>> appliedChanges,
                                                                                    long operatorUid,
                                                                                    UidAndClientId uidAndClientId) {
        //Установка начальных цен на поиске и в сети
        bidService.setPhrasesInitialPrices(uidAndClientId.getClientId(), operatorUid,
                mapList(appliedChanges, ac -> {
                    var option = newSetPhrasesInitialPricesOption();
                    option.setCampaignId(ac.getModel().getId());
                    option.setCampaignCurrency(ac.getModel().getCurrency());
                    DbStrategy newStrategy = ac.getNewValue(TextCampaignWithCustomStrategy.STRATEGY);
                    DbStrategy oldStrategy = ac.getOldValue(TextCampaignWithCustomStrategy.STRATEGY);
                    if (newStrategy.isDifferentPlaces()) {
                        option.setSetPriceContext(true);
                    }
                    // ставим минимальную ставку, если просто включили показы в сети
                    if (newStrategy.getStrategyName() == StrategyName.DEFAULT_
                            && oldStrategy.getStrategyName() == StrategyName.DEFAULT_
                            && !newStrategy.isSearchStop() && !newStrategy.isNetStop()
                            && !newStrategy.isDifferentPlaces()) {
                        option.setSetMinPriceContext(true);
                    }
                    //если кампания создаётся с отключенным поиском, то в ней нет цен на поиске. выставляем их
                    if (!newStrategy.isSearchStop() && (oldStrategy.isAutoBudget() || oldStrategy.isSearchStop())) {
                        option.setSetPrice(true);
                        Boolean calculateStatisticsByType = CALCULATING_BS_STATISTICS_STATUS_BY_CAMPAIGN_TYPE
                                .getOrDefault(ac.getModel().getType(), DEFAULT_CALCULATING_BS_STATISTICS_STATUS);
                        option.setCalculateBsStatisticsFirstPosition(calculateStatisticsByType);
                    }
                    return option;
                })
        );
        //Обнуленяет цены на сеть у ключевых фраз и автотаргетинга для всех кампании.
        bidService.resetPriceContext(operatorUid, filterAndMapList(appliedChanges,
                CampaignStrategyUtils::filterDifferentToSearch,
                AppliedChanges::getModel));
    }

    private static SetPhrasesInitialPricesOption newSetPhrasesInitialPricesOption() {
        return new SetPhrasesInitialPricesOption()
                .withSetPrice(false)
                .withSetPriceContext(false)
                .withSetMinPriceContext(false)
                .withCalculateBsStatisticsFirstPosition(DEFAULT_CALCULATING_BS_STATISTICS_STATUS);
    }

    /**
     * Для ретаргетинга и автотаргетинга: поправим нулевые ставки при переходе с автобюджета на ручное управление
     */
    private <C extends CampaignWithDefaultPriceRecalculation> void setBidsRetargeting(List<AppliedChanges<C>> appliedChanges,
                                                                                      CampaignStrategyChangingSettings settings,
                                                                                      long operatorUid,
                                                                                      UidAndClientId uidAndClientId) {
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        int shard = shardHelper.getShardByClientIdStrictly(uidAndClientId.getClientId());
        List<Retargeting> retargetings = retargetingRepository.getRetargetingsByCampaigns(shard, campaignIds);
        Map<Long, List<Retargeting>> retargetingsMap = StreamEx.of(retargetings)
                .groupingBy(Retargeting::getCampaignId);
        Map<Long, CampaignWithDefaultPriceRecalculation> campaignsMap = listToMap(appliedChanges,
                ac -> ac.getModel().getId(),
                AppliedChanges::getModel);
        Map<Long, AppliedChanges<C>> changesMap = listToMap(appliedChanges,
                ac -> ac.getModel().getId());
        List<AppliedChanges<Retargeting>> retargetingToUpdate = new ArrayList<>();
        List<LogPriceData> priceDataList = new ArrayList<>();

        for (Map.Entry<Long, List<Retargeting>> retargetingEntry : retargetingsMap.entrySet()) {
            AppliedChanges<C> ac = changesMap.get(retargetingEntry.getKey());
            DbStrategy newStrategy = ac.getNewValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
            DbStrategy oldStrategy = ac.getOldValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
            if (newStrategy.isAutoBudget() || oldStrategy == null || !oldStrategy.isAutoBudget()) {
                continue;
            }
            CampaignWithDefaultPriceRecalculation campaign = campaignsMap.get(retargetingEntry.getKey());

            BigDecimal minPrice = campaign.getCurrency().getCurrency().getMinPrice();
            BigDecimal maxPrice = campaign.getCurrency().getCurrency().getMaxPrice();
            for (Retargeting retargeting : retargetingEntry.getValue()) {
                //в маппере 0 заменяется на null. Восстанавливаем значение из базы
                BigDecimal oldPrice = nvl(retargeting.getPriceContext(), BigDecimal.ZERO);
                BigDecimal newPrice = min(max(minPrice, oldPrice), maxPrice);
                if (newPrice.compareTo(oldPrice) != 0) {
                    LogPriceData logPriceData = calculateLogPriceData(campaign, newPrice,
                            retargeting.getAdGroupId(), retargeting.getId(), BigDecimal.ZERO,
                            LogPriceData.OperationType.UPDATE_ZERO_CONTEXT_PRICES);

                    priceDataList.add(logPriceData);
                }
                retargetingToUpdate.add(applyRetargetingPriceChanges(retargeting, newPrice));
            }
        }
        retargetingRepository.updateRetargetings(shard, retargetingToUpdate);
        logPriceService.logPrice(priceDataList, operatorUid);
        relevanceMatchBidsResyncAndFix(shard, settings, appliedChanges, operatorUid);
    }

    public <C extends CampaignWithStrategy> void relevanceMatchBidsResyncAndFix(int shard,
                                                                                CampaignStrategyChangingSettings settings,
                                                                                List<AppliedChanges<C>> appliedChanges,
                                                                                long operatorUid) {

        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        Map<Long, C> campaignsMap = listToMap(appliedChanges,
                ac -> ac.getModel().getId(),
                AppliedChanges::getModel);
        Map<Long, AppliedChanges<C>> changesMap = listToMap(appliedChanges,
                ac -> ac.getModel().getId());


        List<Bid> relevanceMatches = bidRepository.getRelevanceMatchByCampaignIdsNotDeleted(shard, campaignIds);
        List<Bid> relevanceMatchesOnlySync = new ArrayList<>();
        List<Bid> relevanceMatchesNeedPrice = new ArrayList<>();
        List<Bid> relevanceMatchesNeedPriceContext = new ArrayList<>();
        for (Bid rm : relevanceMatches) {
            AppliedChanges<C> ac = changesMap.get(rm.getCampaignId());
            DbStrategy newStrategy = ac.getNewValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
            DbStrategy oldStrategy = ac.getOldValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
            if ((newStrategy.isAutoBudget() || oldStrategy == null || !oldStrategy.isAutoBudget())
                    && (
                    !newStrategy.isDifferentPlaces()
                            || (oldStrategy != null && oldStrategy.isDifferentPlaces()))
                    && !isStopChanged(newStrategy, oldStrategy)) {
                continue;
            }
            C campaignWithStrategy = campaignsMap.get(rm.getCampaignId());
            BigDecimal minPrice = settings.getMinPrice();
            BigDecimal maxPrice = settings.getMaxPrice();

            boolean rmNeedPriceCorrection = rm.getPrice().compareTo(minPrice) < 0 ||
                    rm.getPrice().compareTo(maxPrice) > 0;
            boolean rmNeedCtxPriceCorrection =
                    settings.isHasExtendedRelevanceMatch() && isCtxPriceCorrectionAllowed(campaignWithStrategy.getStrategy())
                            && (rm.getPriceContext().compareTo(minPrice) < 0 || rm.getPriceContext().compareTo(maxPrice) > 0);
            if (!rmNeedPriceCorrection && !rmNeedCtxPriceCorrection) {
                relevanceMatchesOnlySync.add(rm);
                continue;
            }
            if (rmNeedPriceCorrection) {
                relevanceMatchesNeedPrice.add(rm);
            }
            if (rmNeedCtxPriceCorrection) {
                relevanceMatchesNeedPriceContext.add(rm);
            }

        }
        if (!relevanceMatchesOnlySync.isEmpty()) {
            List<LogPriceData> priceDataList = new ArrayList<>();
            for (Bid bid : relevanceMatchesOnlySync) {
                LogPriceData logPriceData = calculateLogPriceData(campaignsMap.get(bid.getCampaignId()),
                        bid.getPriceContext(), bid.getAdGroupId(), bid.getId(),
                        bid.getPrice(), LogPriceData.OperationType.RESTORE_MANUAL_PRICES);
                priceDataList.add(logPriceData);
            }
            logPriceService.logPrice(priceDataList, operatorUid);
            bidRepository.resetBidsBaseSyncedQuet(shard, mapList(relevanceMatchesOnlySync, Bid::getId));
        }

        if (relevanceMatchesNeedPrice.isEmpty() && relevanceMatchesNeedPriceContext.isEmpty()) {
            return;
        }

        Set<Long> relevanceMatchPids =
                relevanceMatchesNeedPrice.stream().map(Bid::getAdGroupId).collect(Collectors.toSet());
        relevanceMatchPids.addAll(relevanceMatchesNeedPriceContext.stream().map(Bid::getAdGroupId).collect(Collectors.toSet()));

        Map<Long, List<Keyword>> keywords = keywordRepository.getKeywordsByAdGroupIds(shard, relevanceMatchPids);
        if (!relevanceMatchesNeedPrice.isEmpty()) {
            Map<Long, List<BigDecimal>> pid2prices = keywords.entrySet()
                    .stream()
                    .collect(Collectors.toMap(
                            Map.Entry::getKey,
                            e -> mapList(e.getValue(), Keyword::getPrice)
                    ));
            updateRelevanceMatchPrice(shard, campaignsMap, relevanceMatchesNeedPrice, pid2prices, operatorUid);
        }

        if (!relevanceMatchesNeedPriceContext.isEmpty()) {
            Map<Long, List<ProcessedAuctionStat>> clicks = bsAuctionStatRepository.getAdGroupStat(shard,
                    mapList(relevanceMatchesNeedPriceContext, Bid::getAdGroupId)
            );
            Map<Long, List<CommonCampaignPriceRecalculationService.PriceAndClicks>> pid2prices = keywords.entrySet()
                    .stream()
                    .collect(Collectors.toMap(
                            Map.Entry::getKey,
                            e -> collectPrices(e.getValue(), clicks.get(e.getKey()))
                    ));
            updateRelevanceMatchPriceContext(shard, campaignsMap, relevanceMatchesNeedPriceContext, pid2prices,
                    operatorUid);
        }
    }

    private static List<CommonCampaignPriceRecalculationService.PriceAndClicks> collectPrices(List<Keyword> keywords,
                                                                                              List<ProcessedAuctionStat> clicks) {
        Map<BigInteger, Long> keywordClicks = listToMap(
                nvl(clicks, Collections.emptyList()),
                ProcessedAuctionStat::getPhraseId, ProcessedAuctionStat::getClicks);
        return mapList(
                keywords,
                k -> new CommonCampaignPriceRecalculationService.PriceAndClicks(getPrice(k.getPriceContext(),
                        k.getPrice()),
                        keywordClicks.getOrDefault(k.getPhraseBsId(), 0L)));
    }

    private <C extends CampaignWithStrategy> void updateRelevanceMatchPriceContext(
            int shard,
            Map<Long, C> campaignsMap,
            List<Bid> relevanceMatchesNeedPrice,
            Map<Long, List<CommonCampaignPriceRecalculationService.PriceAndClicks>> pid2pricesAndClicks,
            long operatorUid) {
        List<LogPriceData> priceDataList = new ArrayList<>();
        List<AppliedChanges<Bid>> relevanceMatchesPriceData = new ArrayList<>();
        for (Bid bid : relevanceMatchesNeedPrice) {
            C campaignWithStrategy = campaignsMap.get(bid.getCampaignId());
            LogPriceData logPriceData = calculateLogPriceData(campaignWithStrategy,
                    bid.getPriceContext(), bid.getAdGroupId(), bid.getId(),
                    bid.getPrice(), LogPriceData.OperationType.RESTORE_MANUAL_PRICES);

            List<CommonCampaignPriceRecalculationService.PriceAndClicks> priceAndClicks =
                    pid2pricesAndClicks.get(bid.getAdGroupId());
            if (priceAndClicks != null) {
                BigDecimal newPrice = calcAveragePrice(priceAndClicks, campaignWithStrategy.getCurrency());
                logPriceData.withPriceCtx(newPrice.doubleValue());
                ModelChanges<Bid> changes = new ModelChanges<>(bid.getId(), Bid.class);
                changes.process(StatusBsSynced.NO, Bid.STATUS_BS_SYNCED);
                changes.process(newPrice, Bid.PRICE_CONTEXT);
                AppliedChanges<Bid> appliedChanges = changes.applyTo(bid);
                relevanceMatchesPriceData.add(appliedChanges);
            } else {
                ModelChanges<Bid> changes = new ModelChanges<>(bid.getId(), Bid.class);
                changes.process(StatusBsSynced.NO, Bid.STATUS_BS_SYNCED);
                changes.process(getDefaultPrice(campaignWithStrategy), Bid.PRICE_CONTEXT);
                AppliedChanges<Bid> appliedChanges = changes.applyTo(bid);
                relevanceMatchesPriceData.add(appliedChanges);
                logPriceData.withPriceCtx(getDefaultPrice(campaignWithStrategy).doubleValue());
            }
            priceDataList.add(logPriceData);
        }
        logPriceService.logPrice(priceDataList, operatorUid);
        bidRepository.setBidsInBidsBase(shard, relevanceMatchesPriceData);
    }

    private static boolean isStopChanged(DbStrategy newStrategy, DbStrategy oldStrategy) {
        if (oldStrategy == null) {
            return newStrategy.isNetStrategy() || newStrategy.isSearchStrategy();
        }
        if (newStrategy.isSearchStrategy()) {
            return oldStrategy.isSearchStop();
        }
        return newStrategy.isNetStrategy() && oldStrategy.isNetStop();
    }

    private static class PriceAndClicks {
        public PriceAndClicks(BigDecimal price, Long clicks) {
            this.price = price;
            this.clicks = clicks;
        }

        private BigDecimal price;
        private Long clicks;

        public BigDecimal getPrice() {
            return price;
        }

        public void setPrice(BigDecimal price) {
            this.price = price;
        }

        public Long getClicks() {
            return clicks;
        }

        public void setClicks(Long clicks) {
            this.clicks = clicks;
        }
    }

    private static BigDecimal calcAveragePrice(List<CommonCampaignPriceRecalculationService.PriceAndClicks> priceAndClicks,
                                               CurrencyCode currency) {
        BigDecimal priceSum = BigDecimal.ZERO;
        BigDecimal clicksSum = BigDecimal.ZERO;
        for (CommonCampaignPriceRecalculationService.PriceAndClicks priceAndClick : priceAndClicks) {
            BigDecimal clicksMultiplier = new BigDecimal(nvl(priceAndClick.getClicks(), 0L) + 1);
            priceSum = priceSum.add(nvl(priceAndClick.getPrice(), BigDecimal.ZERO).multiply(clicksMultiplier));
            clicksSum = clicksSum.add(clicksMultiplier);
        }
        return currencyPriceRounding(priceSum.divide(clicksSum, SCALE_FOR_MONEY, ROUNDING_MODE), currency);
    }

    private static BigDecimal currencyPriceRounding(BigDecimal price, CurrencyCode currency) {
        price = max(price, currency.getCurrency().getMinPrice());
        price = min(price, currency.getCurrency().getMaxPrice());
        return MoneyUtils.roundToAuctionStep(price, currency.getCurrency(), RoundingMode.HALF_UP);
    }

    private static BigDecimal getPrice(BigDecimal priceContext, BigDecimal price) {
        if (priceContext == null) {
            return price;
        }
        if (priceContext.compareTo(BigDecimal.ZERO) == 0) {
            return price;
        }
        return priceContext;
    }

    private <C extends CampaignWithStrategy> void updateRelevanceMatchPrice(int shard,
                                                                            Map<Long, C> campaignsMap,
                                                                            List<Bid> relevanceMatchesNeedPrice,
                                                                            Map<Long, List<BigDecimal>> pid2prices,
                                                                            long operatorUid) {
        List<LogPriceData> priceDataList = new ArrayList<>();
        List<AppliedChanges<Bid>> relevanceMatchesPriceData = new ArrayList<>();
        for (Bid bid : relevanceMatchesNeedPrice) {
            C campaignWithStrategy = campaignsMap.get(bid.getCampaignId());

            LogPriceData logPriceData = calculateLogPriceData(campaignWithStrategy,
                    bid.getPriceContext(), bid.getAdGroupId(), bid.getId(),
                    bid.getPrice(), LogPriceData.OperationType.RESTORE_MANUAL_PRICES);

            List<BigDecimal> price = pid2prices.get(bid.getAdGroupId());
            if (price != null && !price.isEmpty()) {
                BigDecimal newPrice = calcPrice(price, campaignWithStrategy.getCurrency());
                logPriceData.withPrice(newPrice.doubleValue());
                ModelChanges<Bid> changes = new ModelChanges<>(bid.getId(), Bid.class);
                changes.process(StatusBsSynced.NO, Bid.STATUS_BS_SYNCED);
                changes.process(newPrice, Bid.PRICE);
                AppliedChanges<Bid> appliedChanges = changes.applyTo(bid);
                relevanceMatchesPriceData.add(appliedChanges);
            } else {
                ModelChanges<Bid> changes = new ModelChanges<>(bid.getId(), Bid.class);
                changes.process(StatusBsSynced.NO, Bid.STATUS_BS_SYNCED);
                changes.process(getDefaultPrice(campaignWithStrategy), Bid.PRICE);
                AppliedChanges<Bid> appliedChanges = changes.applyTo(bid);
                relevanceMatchesPriceData.add(appliedChanges);
                logPriceData.withPrice(getDefaultPrice(campaignWithStrategy).doubleValue());
            }
            priceDataList.add(logPriceData);
        }
        logPriceService.logPrice(priceDataList, operatorUid);
        bidRepository.setBidsInBidsBase(shard, relevanceMatchesPriceData);
    }

    private static BigDecimal calcPrice(List<BigDecimal> prices, CurrencyCode currency) {
        return currencyPriceRounding(calcPercentile(prices, new BigDecimal("0.3")), currency);
    }

    static BigDecimal calcPercentile(List<BigDecimal> prices, BigDecimal percentile) {
        List<BigDecimal> sortedPrices = mapList(prices, p -> nvl(p, BigDecimal.ZERO));
        Collections.sort(sortedPrices);
        BigDecimal percentileIndex = percentile.multiply(new BigDecimal(sortedPrices.size() - 1));
        int lowIndex = percentileIndex.setScale(0, RoundingMode.FLOOR).intValue();
        int highIndex = percentileIndex.setScale(0, RoundingMode.CEILING).intValue();
        if (lowIndex == highIndex) {
            return sortedPrices.get(lowIndex);
        }
        BigDecimal lowValue = sortedPrices.get(lowIndex).multiply(new BigDecimal(highIndex).subtract(percentileIndex));
        BigDecimal highValue = sortedPrices.get(highIndex).multiply(percentileIndex.subtract(new BigDecimal(lowIndex)));
        return lowValue.add(highValue);
    }

    private static boolean isCtxPriceCorrectionAllowed(DbStrategy strategy) {
        return strategy.isDifferentPlaces() && strategy.isNetStrategy();
    }

    private static BigDecimal getDefaultPrice(CommonCampaign campaignWithStrategy) {
        return campaignWithStrategy.getCurrency().getCurrency().getDefaultPrice();
    }

    /**
     * Восстановление ручных ставок
     */
    public <C extends CampaignWithStrategy> void restoreManualBids(List<AppliedChanges<C>> appliedChanges,
                                                                   CampaignStrategyChangingSettings settings,
                                                                   UidAndClientId uidAndClientId) {
        bidService.restoreManualBids(uidAndClientId.getClientId(), settings,
                mapList(appliedChanges, AppliedChanges::getModel));
    }

    /**
     * очистить прогноз
     */
    public <C extends CampaignWithStrategy> void cleanAutobudgetForecast(List<AppliedChanges<C>> appliedChanges,
                                                                         UidAndClientId uidAndClientId) {
        autobudgetForecastRepository.delete(shardHelper.getShardByClientId(uidAndClientId.getClientId()),
                mapList(appliedChanges, ac -> ac.getModel().getId()));
    }

    /**
     * обновляем ставки, если переходим на автобюджет с неавтобюджетной стратегии
     * ИЛИ если переходим с поискового автобюджета на контекстный или наоборот
     */
    private <C extends CampaignWithDefaultPriceRecalculation> void enableAutoBudget(List<AppliedChanges<C>> appliedChanges,
                                                                                    long operatorUid,
                                                                                    UidAndClientId uidAndClientId) {
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        if (campaignIds.isEmpty()) {
            return;
        }
        int shard = shardHelper.getShardByClientId(uidAndClientId.getClientId());
        scheduleForecast(shard, campaignIds);
        //включаем автобюджет и сбрасываем установленные ранее отметки о приостановке показов,
        //если переходим на автобюджет с неавтобюджетной стратегии
        List<Long> campaignIdsFromManual = filterAndMapList(appliedChanges,
                ac -> !ac.getOldValue(CampaignWithDefaultPriceRecalculation.STRATEGY).isAutoBudget(),
                ac -> ac.getModel().getId());
        bidService.saveManualBids(campaignService.getCampaigns(uidAndClientId.getClientId(), campaignIdsFromManual));
        adGroupRepository.updateStatusAutoBudgetShowForCampaign(shard, campaignIdsFromManual, StatusAutobudgetShow.YES);
        //оставляем текущие цены, ограниченные сверху максимальной ставкой автобюджета, а снизу минимальной ставкой
        //в дальнейшем автобюджет их подкорректирует
        bidService.updateBidsOnAutoBudget(uidAndClientId.getClientId(), operatorUid,
                mapList(appliedChanges, AppliedChanges::getModel));
        //обновляем autobudget_forecast
        addAutobudgetForecast(shard, campaignIds);
    }

    public void addAutobudgetForecast(int shard, List<Long> campaignIds) {
        autobudgetForecastRepository.addAutobudgetForecast(shard, mapList(campaignIds, cid ->
                new AutobudgetForecast()
                        .withCid(cid)
                        .withAutobudgetForecastClicks(0L)
                        .withAutobudgetForecast(0.0)
                        .withStatusAutobudgetForecast(AutobudgetForecastStatus.NEW)));
    }

    /**
     * Запланировать перерасчёт прогноза
     */
    public void scheduleForecast(int shard, List<Long> campaignIds) {
        campaignRepository.setAutobudgetForecastDate(shard, campaignIds, null);
    }

    public <C extends CampaignWithStrategy> void mailNotification(List<AppliedChanges<C>> appliedChanges,
                                                                  Long operatorUid, UidAndClientId uidAndClientId) {

        List<StrategyEvent> events = new ArrayList<>();
        for (var appliedChange : appliedChanges) {
            DbStrategy newStrategy = appliedChange.getNewValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
            if (newStrategy.getStrategyData() == null) {
                continue;
            }
            CurrencyCode currency = appliedChange.getNewValue(CampaignWithDefaultPriceRecalculation.CURRENCY);
            DbStrategy oldStrategy = appliedChange.getOldValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
            String text = mailTextCreatorService.makeText(oldStrategy, newStrategy, currency);
            EventStrategyParams eventStrategyParams = new EventStrategyParams(text);
            StrategyEvent strategyEvent = new StrategyEvent(appliedChange.getModel().getId(), operatorUid,
                    uidAndClientId.getUid(), appliedChange.getModel().getId(), eventStrategyParams);
            events.add(strategyEvent);
        }
        mailNotificationEventService.queueEvents(operatorUid, uidAndClientId.getClientId(), events);
    }

    public <C extends CampaignWithStrategy> void markStrategyChange(List<AppliedChanges<C>> appliedChanges,
                                                                    Long operatorUid) {
        campaignService.logStrategyChange(appliedChanges, operatorUid);
    }

    public void resetBidsRetargetingsBsSyncStatus(int shard, List<Long> campaignIds) {
        List<Retargeting> retargetings = retargetingRepository.getRetargetingsByCampaigns(shard, campaignIds);
        retargetingRepository.updateRetargetings(shard, mapList(retargetings, this::applyBidRetargetingReseting));
    }

    private AppliedChanges<Retargeting> applyBidRetargetingReseting(Retargeting retargeting) {
        return new ModelChanges<>(retargeting.getId(), Retargeting.class)
                .process(3, Retargeting.AUTOBUDGET_PRIORITY)
                .process(StatusBsSynced.NO, Retargeting.STATUS_BS_SYNCED)
                .applyTo(retargeting);
    }

}
