package ru.yandex.direct.core.entity.campaign.service.pricerecalculation;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.StatusAutobudgetShow;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteriaSimple;
import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.entity.campaign.container.CampaignStrategyChangingSettings;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithPriceRecalculation;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria.fromShowConditionList;
import static ru.yandex.direct.core.entity.bids.service.BidService.autoStrategyPrice;
import static ru.yandex.direct.core.entity.bids.service.BidService.calculateNewPriceForCpmBidOnDisablingAutobudget;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.AD_GROUP_TYPES_TO_UPDATE_CPM_RETARGETINGS_ON_ENABLING_AUTOBUDGET;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для пересчета ставок при смене стратегий Cpm
 */
@Service
public class CpmCampaignPriceRecalculationService {

    private final ShardHelper shardHelper;
    private final CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService;
    private final BidRepository bidRepository;
    private final LogPriceService logPriceService;
    private final BidService bidService;
    private final AdGroupRepository adGroupRepository;
    private final RetargetingRepository retargetingRepository;

    public CpmCampaignPriceRecalculationService(ShardHelper shardHelper,
                                                BidRepository bidRepository,
                                                LogPriceService logPriceService,
                                                BidService bidService,
                                                AdGroupRepository adGroupRepository,
                                                RetargetingRepository retargetingRepository,
                                                CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService) {
        this.shardHelper = shardHelper;
        this.bidRepository = bidRepository;
        this.logPriceService = logPriceService;
        this.bidService = bidService;
        this.adGroupRepository = adGroupRepository;
        this.retargetingRepository = retargetingRepository;
        this.commonCampaignPriceRecalculationService = commonCampaignPriceRecalculationService;
    }

    /**
     * Действия при изменения стратегии. Массовые действия
     */
    public void afterCampaignsStrategyChanged(List<AppliedChanges<CpmCampaignWithPriceRecalculation>> appliedChanges,
                                              CampaignStrategyChangingSettings settings,
                                              Long operatorUid,
                                              UidClientIdShard uidClientIdShard) {
        campaignsPriceChange(appliedChanges, settings, operatorUid, uidClientIdShard);
        commonCampaignPriceRecalculationService.markStrategyChange(appliedChanges, operatorUid);
        commonCampaignPriceRecalculationService.mailNotification(appliedChanges, operatorUid,
                uidClientIdShard.getUidAndClientId()); //TODO перенести в AdditionalActionsContainer
    }

    /**
     * Пересчет ставок при смене стратегий.
     * Аналог функции campaign_strategy_changed в perl.
     * Обрататывает только CPM кампании.
     */
    private void campaignsPriceChange(List<AppliedChanges<CpmCampaignWithPriceRecalculation>> appliedChanges,
                                      CampaignStrategyChangingSettings settings,
                                      long operatorUid,
                                      UidClientIdShard uidClientIdShard) {
        List<AppliedChanges<CpmCampaignWithPriceRecalculation>> campaignsWithChangingStrategyFromAutoBudgetToManual =
                filterList(appliedChanges, CampaignStrategyUtils::filterAutoBudgetToManual);

        commonCampaignPriceRecalculationService
                .restoreManualBids(campaignsWithChangingStrategyFromAutoBudgetToManual, settings,
                        uidClientIdShard.getUidAndClientId());
        commonCampaignPriceRecalculationService
                .cleanAutobudgetForecast(campaignsWithChangingStrategyFromAutoBudgetToManual,
                        uidClientIdShard.getUidAndClientId());
        enableAutoBudget(filterList(appliedChanges, CampaignStrategyUtils::filterEnableAutoBudget), settings,
                operatorUid, uidClientIdShard.getUidAndClientId());
        updatePhrasesPriceCampaignsOnAutobudgetDisabling(settings, uidClientIdShard.getClientId(), operatorUid,
                campaignsWithChangingStrategyFromAutoBudgetToManual);
        setBidsRetargetingCampaigns(campaignsWithChangingStrategyFromAutoBudgetToManual, settings, operatorUid,
                uidClientIdShard.getUidAndClientId());
        commonCampaignPriceRecalculationService
                .relevanceMatchBidsResyncAndFix(uidClientIdShard.getShard(), settings, appliedChanges, operatorUid);

        List<Long> campaignIdsWithAutobudget = filterAndMapList(appliedChanges,
                CampaignStrategyUtils::filterToAutobudget, c -> c.getModel().getId());
        commonCampaignPriceRecalculationService
                .resetBidsRetargetingsBsSyncStatus(uidClientIdShard.getShard(), campaignIdsWithAutobudget);
    }

    /**
     * Обновляем ставки, если переходим на автобюджет с неавтобюджетной стратегии
     * ИЛИ если переходим с поискового автобюджета на контекстный или наоборот
     */
    private <C extends CpmCampaignWithPriceRecalculation> void enableAutoBudget(
            List<AppliedChanges<C>> appliedChanges,
            CampaignStrategyChangingSettings settings,
            long operatorUid,
            UidAndClientId uidAndClientId) {
        if (appliedChanges.isEmpty()) {
            return;
        }
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        int shard = shardHelper.getShardByClientId(uidAndClientId.getClientId());
        commonCampaignPriceRecalculationService.scheduleForecast(shard, campaignIds);
        //включаем автобюджет и сбрасываем установленные ранее отметки о приостановке показов,
        //если переходим на автобюджет с неавтобюджетной стратегии
        List<Long> campaignIdsFromManual = filterAndMapList(appliedChanges,
                ac -> !ac.getOldValue(CampaignWithStrategy.STRATEGY).isAutoBudget(),
                ac -> ac.getModel().getId());
        bidService.saveManualBidsForCampaignIds(campaignIdsFromManual);
        adGroupRepository.updateStatusAutoBudgetShowForCampaign(shard, campaignIdsFromManual, StatusAutobudgetShow.YES);

        List<C> campaigns = mapList(appliedChanges, AppliedChanges::getModel);
        List<Bid> bids = bidRepository.getBidsByCampaignIds(shard, mapList(campaigns, CampaignWithStrategy::getId));
        Map<Long, CampaignWithStrategy> campaignsMap = listToMap(campaigns, CampaignWithStrategy::getId);
        List<Bid> bidsToBeUpdated = filterList(bids, bid -> checkPriceNeedChange(settings, bid,
                campaignsMap.get(bid.getCampaignId())));

        bidService.updateBidsOnAutoBudgetForCpm(shard, settings, uidAndClientId.getClientId(), operatorUid,
                campaignsMap, bidsToBeUpdated);
        commonCampaignPriceRecalculationService.addAutobudgetForecast(shard, campaignIds);
    }

    private static boolean checkPriceNeedChange(CampaignStrategyChangingSettings settings,
                                                Bid bid,
                                                CampaignWithStrategy campaign) {
        return bid.getPrice().compareTo(BigDecimal.ZERO) != 0
                || bid.getPriceContext().compareTo(autoStrategyPrice(settings, bid.getPriceContext(), campaign)) != 0;
    }

    private <C extends CpmCampaignWithPriceRecalculation> void updatePhrasesPriceCampaignsOnAutobudgetDisabling(
            CampaignStrategyChangingSettings settings,
            ClientId clientId,
            Long operatorUid,
            List<AppliedChanges<C>> appliedChanges) {
        ShowConditionSelectionCriteria showConditionSelectionCriteria =
                fromShowConditionList(mapList(appliedChanges,
                        ac -> BidSelectionCriteriaSimple.fromCampaignId(ac.getModel().getId())))
                        .withAdGroupTypes(Set.of(AdGroupType.CPM_BANNER));
        List<Bid> bids = bidService.getBids(clientId, operatorUid, showConditionSelectionCriteria, maxLimited());
        Map<Long, BigDecimal> oldAvgCpmByCampaignId = listToMap(appliedChanges,
                ac -> ac.getModel().getId(),
                ac -> nvl(ac.getOldValue(CampaignWithStrategy.STRATEGY).getStrategyData().getAvgCpm(),
                        settings.getMinPrice()));
        List<Bid> updatingBids = filterList(bids,
                bid -> bid.getPriceContext().compareTo(
                        calculateNewPriceForCpmBidOnDisablingAutobudget(
                                settings,
                                oldAvgCpmByCampaignId.get(bid.getCampaignId()),
                                bid.getPriceContext())) != 0);

        List<Long> keywordIds = mapList(updatingBids, Bid::getId);
        bidService.updatePhrasesContextPriceForCpmCampaignsOnAutobudgetDisabling(settings, clientId,
                oldAvgCpmByCampaignId,
                keywordIds);
    }

    private <C extends CpmCampaignWithPriceRecalculation> void setBidsRetargetingCampaigns(
            List<AppliedChanges<C>> appliedChanges,
            CampaignStrategyChangingSettings settings,
            long operatorUid, UidAndClientId uidAndClientId) {
        int shard = shardHelper.getShardByClientIdStrictly(uidAndClientId.getClientId());

        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());

        List<Retargeting> retargetings = retargetingRepository.getRetargetingsByCampaignIdsAndAdGroupType(shard,
                campaignIds, AD_GROUP_TYPES_TO_UPDATE_CPM_RETARGETINGS_ON_ENABLING_AUTOBUDGET);
        Map<Long, List<Retargeting>> retargetingsByCampaignId = StreamEx.of(retargetings)
                .groupingBy(Retargeting::getCampaignId);

        Map<Long, DbStrategy> oldStrategyByCampaignId = listToMap(appliedChanges, ac -> ac.getModel().getId(),
                ac -> ac.getOldValue(CpmCampaignWithPriceRecalculation.STRATEGY));

        List<AppliedChanges<Retargeting>> retargetingsToUpdate = getRetargetingsChanges(settings,
                retargetingsByCampaignId,
                oldStrategyByCampaignId);

        Map<Long, C> campaignById = listToMap(appliedChanges, ac -> ac.getModel().getId(),
                AppliedChanges::getModel);

        List<LogPriceData> priceDataList = getRetargetingLogPriceData(settings, retargetingsByCampaignId,
                oldStrategyByCampaignId, campaignById);

        retargetingRepository.updateRetargetings(shard, retargetingsToUpdate);
        logPriceService.logPrice(priceDataList, operatorUid);
    }

    private <C extends CampaignWithStrategy> List<LogPriceData> getRetargetingLogPriceData(
            CampaignStrategyChangingSettings settings,
            Map<Long, List<Retargeting>> retargetingsMap,
            Map<Long, DbStrategy> oldStrategyByCampaignId,
            Map<Long, C> campaignById) {
        return EntryStream.of(retargetingsMap)
                .mapKeyValue((cid, campaignRetargetings) ->
                        getRetargetingLogPriceDataForCampaign(settings, oldStrategyByCampaignId.get(cid),
                                campaignById.get(cid), campaignRetargetings))
                .flatMap(Collection::stream)
                .toList();
    }

    private <C extends CampaignWithStrategy> List<LogPriceData> getRetargetingLogPriceDataForCampaign(
            CampaignStrategyChangingSettings settings,
            DbStrategy strategy,
            C campaign,
            List<Retargeting> campaignRetargetings) {
        return StreamEx.of(campaignRetargetings)
                .mapToEntry(retargeting -> calculateNewPriceForCpmBidOnDisablingAutobudget(settings,
                        nvl(strategy.getStrategyData().getAvgCpm(), settings.getMinPrice()),
                        retargeting.getPriceContext()))
                .filterKeyValue((retargeting, newPrice) -> newPrice.compareTo(nvl(retargeting.getPriceContext(),
                        BigDecimal.ZERO)) == 0)
                .mapKeyValue((retargeting, newPrice) -> commonCampaignPriceRecalculationService.calculateLogPriceData(
                        campaign,
                        newPrice,
                        retargeting.getAdGroupId(), retargeting.getId(), BigDecimal.ZERO,
                        LogPriceData.OperationType.UPDATE_ZERO_CONTEXT_PRICES))
                .toList();
    }


    private List<AppliedChanges<Retargeting>> getRetargetingsChanges(CampaignStrategyChangingSettings settings,
                                                                     Map<Long, List<Retargeting>> retargetingsMap,
                                                                     Map<Long, DbStrategy> oldStrategyByCampaignId) {
        return EntryStream.of(retargetingsMap)
                .mapKeys(oldStrategyByCampaignId::get)
                .mapKeyValue((strategy, campaignRetargetings) -> getRetargetingsChanges(settings, strategy,
                        campaignRetargetings))
                .flatMap(Collection::stream)
                .toList();
    }

    private List<AppliedChanges<Retargeting>> getRetargetingsChanges(CampaignStrategyChangingSettings settings,
                                                                     DbStrategy oldStrategy,
                                                                     List<Retargeting> campaignRetargetings) {
        return StreamEx.of(campaignRetargetings)
                .mapToEntry(retargeting -> calculateNewPriceForCpmBidOnDisablingAutobudget(settings,
                        nvl(oldStrategy.getStrategyData().getAvgCpm(), settings.getMinPrice()),
                        retargeting.getPriceContext()))
                .mapKeyValue(commonCampaignPriceRecalculationService::applyRetargetingPriceChanges)
                .toList();
    }
}
