package ru.yandex.direct.core.entity.campaign.service.pricerecalculation;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.markupcondition.repository.MarkupConditionRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelWithId;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.campaign.service.pricerecalculation.PriceCalculator.getSeasonalPriceRatio;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.getPackagePriceFunctionByCampaignId;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class CpmPriceCampaignPriceRecalculationService {
    private final ShardHelper shardHelper;
    private final PricePackageRepository pricePackageRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final RetargetingRepository retargetingRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final MarkupConditionRepository markupConditionRepository;
    private final CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService;


    public CpmPriceCampaignPriceRecalculationService(ShardHelper shardHelper,
                                                     PricePackageRepository pricePackageRepository,
                                                     CampaignRepository campaignRepository,
                                                     CampaignTypedRepository campaignTypedRepository,
                                                     AdGroupRepository adGroupRepository,
                                                     RetargetingRepository retargetingRepository,
                                                     RetargetingConditionRepository retargetingConditionRepository,
                                                     MarkupConditionRepository markupConditionRepository,
                                                     CommonCampaignPriceRecalculationService
                                                             commonCampaignPriceRecalculationService) {
        this.shardHelper = shardHelper;
        this.pricePackageRepository = pricePackageRepository;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.retargetingRepository = retargetingRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.markupConditionRepository = markupConditionRepository;
        this.commonCampaignPriceRecalculationService = commonCampaignPriceRecalculationService;
    }

    public void afterCpmPriceCampaignsDateChanged(List<AppliedChanges<CampaignWithPricePackage>> appliedChanges,
                                                  UidClientIdShard uidClientIdShard) {
        int shard = uidClientIdShard.getShard();

        List<Long> campaignIds = mapList(appliedChanges, c -> c.getModel().getId());
        List<Retargeting> retargetings = retargetingRepository.getRetargetingsByCampaigns(shard, campaignIds);
        updateRetargetingsPrice(shard, retargetings);
    }

    public boolean needRejectCampaignApprove(PricePackage pricePackage, CampaignWithPricePackage campaignWithPricePackage) {

        CpmPriceCampaign campaign;
        try {
            campaign = (CpmPriceCampaign) campaignWithPricePackage;
        } catch (Exception e) {
            return false;
        }
        var priceMarkup = getSeasonalPriceRatio(campaign, pricePackage);
        return priceMarkup == null;
    }

    private void updateRetargetingsPrice(int shard, List<Retargeting> retargetings) {
        List<Long> campaignIds = mapList(retargetings, Retargeting::getCampaignId);

        var campaignsType = campaignRepository.getCampaignsTypeMap(shard, campaignIds);
        var packagePriceFunctionByCampaignId =
                getPackagePriceFunctionByCampaignId(shard, campaignTypedRepository,
                        pricePackageRepository, markupConditionRepository, campaignsType);

        List<AppliedChanges<Retargeting>> retargetingToUpdate = new ArrayList<>();

        Map<Long, List<Long>> adGroupIdsMap = adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds);
        for (Long campaignId : adGroupIdsMap.keySet()) {
            List<Retargeting> retargetingsForCampaignId = retargetings
                    .stream()
                    .filter(r -> r.getCampaignId().equals(campaignId)).collect(Collectors.toUnmodifiableList());

            List<Long> adGroupIds = mapList(retargetingsForCampaignId, Retargeting::getAdGroupId);
            Map<Long, List<RetargetingCondition>> retConditionsByAdGroupIds =
                    retargetingConditionRepository.getRetConditionsByAdGroupIds(shard, adGroupIds);
            Map<Long, List<Goal>> retargetingGoalsByAdGroupId = EntryStream.of(retConditionsByAdGroupIds)
                    .mapValues(conditions -> conditions.stream()
                            .map(RetargetingCondition::collectGoals)
                            .flatMap(Collection::stream)
                            .distinct()
                            .collect(toList()))
                    .toMap();
            Map<Long, AdGroup> adGroupsMap = listToMap(adGroupRepository.getAdGroups(shard, adGroupIds), AdGroup::getId);

            for (Retargeting retargeting : retargetings) {
                AdGroup adGroup = adGroupsMap.get(retargeting.getAdGroupId());

                List<Long> projectParamConditions = adGroup.getProjectParamConditions();
                List<Long> goalIds = mapList(retargetingGoalsByAdGroupId.get(adGroup.getId()), ModelWithId::getId);
                List<Long> geo = adGroup.getGeo();
                BigDecimal pricePackagePrice = packagePriceFunctionByCampaignId.get(campaignId)
                        .apply(geo, goalIds, projectParamConditions).getPrice();

                retargetingToUpdate.add(commonCampaignPriceRecalculationService
                        .applyRetargetingPriceChanges(retargeting, pricePackagePrice));
            }
        }
        retargetingRepository.updateRetargetings(shard, retargetingToUpdate);
    }
}
