package ru.yandex.direct.core.entity.campaign.service.pricerecalculation;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.bids.model.BidDynamicPrices;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.campaign.container.CampaignStrategyChangingSettings;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaignWithPriceRecalculation;
import ru.yandex.direct.core.entity.campaign.model.TextCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static java.math.BigDecimal.ZERO;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.NumberUtils.equalsByCompareTo;
import static ru.yandex.direct.utils.NumberUtils.greaterThanZero;
import static ru.yandex.direct.utils.NumberUtils.isZero;

/**
 * Сервис для пересчета ставок при смене стратегий ДО
 */
@Service
public class DynamicCampaignPriceRecalculationService {

    private final ShardHelper shardHelper;
    private final BidRepository bidRepository;
    private final LogPriceService logPriceService;
    private final CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService;

    public DynamicCampaignPriceRecalculationService(
            ShardHelper shardHelper,
            BidRepository bidRepository,
            LogPriceService logPriceService,
            CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService) {
        this.shardHelper = shardHelper;
        this.bidRepository = bidRepository;
        this.logPriceService = logPriceService;
        this.commonCampaignPriceRecalculationService = commonCampaignPriceRecalculationService;
    }

    /**
     * Действия при изменения стратегии
     */
    public void afterCampaignsStrategyChanged(
            List<AppliedChanges<DynamicCampaignWithPriceRecalculation>> appliedChanges,
            CampaignStrategyChangingSettings settings,
            Long operatorUid,
            UidClientIdShard uidClientIdShard) {
        campaignsPriceChange(appliedChanges, uidClientIdShard);
        commonCampaignPriceRecalculationService.markStrategyChange(appliedChanges, operatorUid);
        commonCampaignPriceRecalculationService
                .mailNotification(appliedChanges, operatorUid, uidClientIdShard.getUidAndClientId());
    }

    /**
     * Пересчет ставок при смене стратегий.
     * Аналог функции campaign_strategy_changed в perl (только для DYNAMIC кампании).
     */
    private void campaignsPriceChange(List<AppliedChanges<DynamicCampaignWithPriceRecalculation>> appliedChanges,
                                      UidClientIdShard uidClientIdShard) {
        commonCampaignPriceRecalculationService.cleanAutobudgetForecast(filterList(appliedChanges,
                CampaignStrategyUtils::filterAutoBudgetToManual),
                uidClientIdShard.getUidAndClientId());
        setAutobudgetForecast(filterList(appliedChanges, CampaignStrategyUtils::filterEnableAutoBudget),
                uidClientIdShard.getUidAndClientId());

        adjustBidsDynamicPrices(filterList(appliedChanges, CampaignStrategyUtils::filterToManual),
                uidClientIdShard.getUidAndClientId());
        resetBidsDynamicBsStatusAndPriority(filterList(appliedChanges, CampaignStrategyUtils::filterToAutobudget),
                uidClientIdShard.getUidAndClientId());
    }

    private <C extends DynamicCampaignWithPriceRecalculation> void setAutobudgetForecast(
            List<AppliedChanges<C>> appliedChanges,
            UidAndClientId uidAndClientId) {
        if (appliedChanges.isEmpty()) {
            return;
        }
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        int shard = shardHelper.getShardByClientId(uidAndClientId.getClientId());
        commonCampaignPriceRecalculationService.scheduleForecast(shard, campaignIds);

        commonCampaignPriceRecalculationService.addAutobudgetForecast(shard, campaignIds);
    }

    /**
     * Для условий нацеливания: проверим и, если нужно, поправим ставки или приоритет автобюджета
     */
    <C extends DynamicCampaignWithPriceRecalculation> void adjustBidsDynamicPrices(
            List<AppliedChanges<C>> appliedChanges,
            UidAndClientId uidAndClientId) {
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        if (campaignIds.isEmpty()) {
            return;
        }

        int shard = shardHelper.getShardByClientIdStrictly(uidAndClientId.getClientId());
        Map<Long, List<BidDynamicPrices>> campaignIdToBidsDynamic =
                bidRepository.getBidsDynamicPricesByCampaignIds(shard, campaignIds);
        if (campaignIdToBidsDynamic.isEmpty()) {
            return;
        }

        List<AppliedChanges<BidDynamicPrices>> bidsDynamicPricesToUpdate =
                getBidsDynamicPricesToUpdate(appliedChanges, campaignIdToBidsDynamic);
        List<LogPriceData> priceDataList =
                getPriceDataList(appliedChanges, campaignIdToBidsDynamic, bidsDynamicPricesToUpdate);

        logPriceService.logPrice(priceDataList, uidAndClientId.getUid());
        bidRepository.updateBidsDynamicPrices(shard, bidsDynamicPricesToUpdate);
    }

    private <C extends DynamicCampaignWithPriceRecalculation> List<LogPriceData> getPriceDataList(
            List<AppliedChanges<C>> appliedChanges,
            Map<Long, List<BidDynamicPrices>> campaignIdToBidsDynamic,
            List<AppliedChanges<BidDynamicPrices>> bidsDynamicPricesToUpdate) {

        Map<Long, C> dynIdToCampaign = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .mapToEntry(campaign -> campaignIdToBidsDynamic.get(campaign.getId()))
                .nonNullValues()
                .flatMapValues(List::stream)
                .mapValues(BidDynamicPrices::getId)
                .invert()
                .toMap();

        return StreamEx.of(bidsDynamicPricesToUpdate)
                .map(AppliedChanges::getModel)
                .mapToEntry(bid -> dynIdToCampaign.get(bid.getId()))
                .mapKeyValue(this::getLogPriceData)
                .toList();
    }

    private <C extends DynamicCampaignWithPriceRecalculation> List<AppliedChanges<BidDynamicPrices>> getBidsDynamicPricesToUpdate(
            List<AppliedChanges<C>> appliedChanges,
            Map<Long, List<BidDynamicPrices>> campaignIdToBidsDynamic) {
        return bidByCampaign(appliedChanges, campaignIdToBidsDynamic)
                .mapKeyValue(this::getBidDynamicPricesAppliedChanges)
                .filter(this::isAnyPricesHaveChanged)
                .toList();
    }

    private boolean isAnyPricesHaveChanged(AppliedChanges<BidDynamicPrices> appliedBidChanges) {
        BigDecimal newPrice = appliedBidChanges.getNewValue(BidDynamicPrices.PRICE);
        BigDecimal newPriceContext = appliedBidChanges.getNewValue(BidDynamicPrices.PRICE_CONTEXT);
        BigDecimal oldPrice = appliedBidChanges.getOldValue(BidDynamicPrices.PRICE);
        BigDecimal oldPriceContext = appliedBidChanges.getOldValue(BidDynamicPrices.PRICE_CONTEXT);
        return !equalsByCompareTo(newPrice, oldPrice) || !equalsByCompareTo(newPriceContext, oldPriceContext);
    }

    private <C extends DynamicCampaignWithPriceRecalculation> EntryStream<AppliedChanges<C>, BidDynamicPrices> bidByCampaign(
            List<AppliedChanges<C>> appliedChanges,
            Map<Long, List<BidDynamicPrices>> campaignIdToBidsDynamic) {
        return StreamEx.of(appliedChanges)
                .mapToEntry(AppliedChanges::getModel)
                .mapValues(DynamicCampaignWithPriceRecalculation::getId)
                .mapValues(campaignIdToBidsDynamic::get)
                .nonNullValues()
                .flatMapValues(Collection::stream);
    }

    private <C extends DynamicCampaignWithPriceRecalculation> AppliedChanges<BidDynamicPrices> getBidDynamicPricesAppliedChanges(
            AppliedChanges<C> appliedChanges,
            BidDynamicPrices bid) {
        DynamicCampaignWithPriceRecalculation campaign = appliedChanges.getModel();
        DbStrategy newStrategy = appliedChanges.getNewValue(TextCampaignWithCustomStrategy.STRATEGY);
        BigDecimal newDynamicPrice = getNewDynamicPrice(newStrategy,
                campaign.getCurrency().getCurrency(), bid.getPrice(), bid.getPriceContext());
        BigDecimal newDynamicPriceContext = getNewDynamicPriceContext(newStrategy,
                campaign.getCurrency().getCurrency(), bid.getPrice(), bid.getPriceContext(),
                bid.getContextPriceCoef());
        return applyBidsDynamicPriceChanges(bid, newDynamicPrice, newDynamicPriceContext);
    }

    private <C extends DynamicCampaignWithPriceRecalculation> LogPriceData getLogPriceData(BidDynamicPrices bid,
                                                                                           C campaign) {
        return new LogPriceData(
                campaign.getId(),
                bid.getAdGroupId(),
                bid.getId(),
                bid.getPriceContext().doubleValue(),
                bid.getPrice().doubleValue(),
                campaign.getCurrency(),
                LogPriceData.OperationType.UPDATE_2
        );
    }

    private AppliedChanges<BidDynamicPrices> applyBidsDynamicPriceChanges(BidDynamicPrices bidDynamicPrices,
                                                                          BigDecimal newPrice,
                                                                          BigDecimal newPriceContext) {
        return new ModelChanges<>(bidDynamicPrices.getId(), BidDynamicPrices.class)
                .process(newPrice, BidDynamicPrices.PRICE)
                .process(newPriceContext, BidDynamicPrices.PRICE_CONTEXT)
                .applyTo(bidDynamicPrices);
    }

    <C extends DynamicCampaignWithPriceRecalculation> void resetBidsDynamicBsStatusAndPriority(
            List<AppliedChanges<C>> appliedChanges,
            UidAndClientId uidAndClientId) {
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        if (campaignIds.isEmpty()) {
            return;
        }
        int shard = shardHelper.getShardByClientIdStrictly(uidAndClientId.getClientId());
        bidRepository.resetBidsDynamicBsStatusAndPriority(shard, campaignIds);
    }

    BigDecimal getNewDynamicPrice(DbStrategy strategy,
                                  Currency currency,
                                  BigDecimal price,
                                  BigDecimal priceContext) {
        if (strategy.isSearchStop()) {
            return price;
        }
        if (!greaterThanZero(price) && !greaterThanZero(priceContext)) {
            return limitPriceOfCurrency(currency.getDefaultPrice(), currency);
        }
        if (isZero(price)) {
            return limitPriceOfCurrency(priceContext, currency);
        }
        return limitPriceOfCurrency(price, currency);
    }

    BigDecimal getNewDynamicPriceContext(DbStrategy strategy,
                                         Currency currency,
                                         BigDecimal price,
                                         BigDecimal priceContext,
                                         Long contextPriceCoef) {
        if (!strategy.isDifferentPlaces() || strategy.isNetStop()) {
            return ZERO;
        }
        if (!isZero(priceContext)) {
            return limitPriceOfCurrency(priceContext, currency);
        }
        if (!greaterThanZero(price)) {
            return limitPriceOfCurrency(currency.getDefaultPrice(), currency);
        }
        BigDecimal newPriceContext = (BigDecimal.valueOf(contextPriceCoef).multiply(price))
                .divide(BigDecimal.valueOf(100L), 2, RoundingMode.CEILING);
        return limitPriceOfCurrency(newPriceContext, currency);
    }

    private BigDecimal limitPriceOfCurrency(@Nullable BigDecimal price, Currency currency) {
        return price == null ? BigDecimal.ZERO : currency.getMinPrice().max(price.min(currency.getMaxPrice()));
    }
}
