package ru.yandex.direct.core.entity.campaign.service.pricerecalculation;

import java.util.List;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.SmartCampaignWithPriceRecalculation;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils;
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для пересчета ставок при смене стратегий Смарт(performance)
 */
@Service
public class SmartCampaignPriceRecalculationService {

    private final PerformanceFilterRepository performanceFilterRepository;

    public SmartCampaignPriceRecalculationService(PerformanceFilterRepository performanceFilterRepository) {
        this.performanceFilterRepository = performanceFilterRepository;
    }

    /**
     * Действия при изменении стратегии
     * Аналог функции campaign_strategy_changed в perl (только для SMART/PERFORMANCE кампаний).
     */
    public <C extends SmartCampaignWithPriceRecalculation> void afterCampaignsStrategyChanged(
            List<AppliedChanges<C>> appliedChanges,
            UidClientIdShard uidClientIdShard) {
        resetBidsPerformanceBsStatusAndPriority(
                filterList(appliedChanges, CampaignStrategyUtils::filterToAutobudgetRoi), uidClientIdShard);
        resetBidsPerformanceBsStatus(appliedChanges, uidClientIdShard);
    }

    /**
     * при переходе на ROI просто выставим autobudgetPriority на фильтрах в значение по умолчанию
     */
    private <C extends SmartCampaignWithPriceRecalculation> void resetBidsPerformanceBsStatusAndPriority(
            List<AppliedChanges<C>> appliedChanges,
            UidClientIdShard uidClientIdShard) {
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        if (campaignIds.isEmpty()) {
            return;
        }
        performanceFilterRepository.resetBidsPerformanceBsStatusAndPriority(uidClientIdShard.getShard(), campaignIds);
    }

    /**
     * сбрасываем bsStatus у фильтров при любом изменениии стратегии
     */
    private <C extends SmartCampaignWithPriceRecalculation> void resetBidsPerformanceBsStatus(
            List<AppliedChanges<C>> appliedChanges,
            UidClientIdShard uidClientIdShard) {
        List<Long> campaignIds = mapList(appliedChanges, ac -> ac.getModel().getId());
        if (campaignIds.isEmpty()) {
            return;
        }
        performanceFilterRepository.resetBidsPerformanceBsStatus(uidClientIdShard.getShard(), campaignIds);
    }
}
