package ru.yandex.direct.core.entity.campaign.service.type.add;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.AddServicedCampaignInfo;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class AddServicedCampaignService {

    private final ClientService clientService;
    private final RbacService rbacService;

    @Autowired
    public AddServicedCampaignService(ClientService clientService,
                                      RbacService rbacService) {
        this.clientService = clientService;
        this.rbacService = rbacService;
    }

    public <T extends BaseCampaign> List<AddServicedCampaignInfo> getServicedInfoForNewCampaigns(
            CampaignValidationContainer parameters,
            List<T> campaigns) {
        Set<Boolean> requiredUnservice = listToSet(campaigns, campaign -> getRequiredUnservice(parameters, campaign));

        Map<Boolean, AddServicedCampaignInfo> requiredUnserviceToInfo = StreamEx.of(requiredUnservice)
                .mapToEntry(x -> getServicedInfoForClientCampaigns(parameters, x))
                .toMap();

        return mapList(campaigns, campaign -> requiredUnserviceToInfo.get(getRequiredUnservice(parameters, campaign)));
    }

    /**
     * Запросил ли пользователь явно "несервисируемую" кампанию
     *
     * @return true если пользователь запросил "несервисируемую" кампанию (передал REQURE_SERVICING = FALSE)
     * false иначе
     */
    private <T extends BaseCampaign> boolean getRequiredUnservice(
            CampaignValidationContainer parameters,
            T campaign) {
        return parameters.getRequireServicing(campaign) == Boolean.FALSE;
    }

    /**
     * Возвращает информацию о сервисируемости для кампаний клиента
     */
    public AddServicedCampaignInfo getServicedInfoForClientCampaigns(
            CampaignValidationContainer parameters) {
        return getServicedInfoForClientCampaigns(parameters, false);
    }

    private AddServicedCampaignInfo getServicedInfoForClientCampaigns(
            CampaignValidationContainer parameters,
            boolean requiredUnservice) {
        RbacRole operatorRole = rbacService.getUidRole(parameters.getOperatorUid());
        AddServicedCampaignInfo result = new AddServicedCampaignInfo()
                .withIsServiced(false);

        if (operatorRole == RbacRole.AGENCY || rbacService.isUnderAgency(parameters.getChiefUid())) {
            return result;
        }

        //В тирной схеме кампания сервисируется только главным менеджером: DIRECT-116206
        Client client = Objects.requireNonNull(clientService.getClient(parameters.getClientId()));
        if (client.getIsIdmPrimaryManager() && !requiredUnservice) {
            return result.withIsServiced(true)
                    .withManagerUid(client.getPrimaryManagerUid());
        }

        return result;
    }

}
