package ru.yandex.direct.core.entity.campaign.service.type.add;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.brandSurvey.BrandSurvey;
import ru.yandex.direct.core.entity.campaign.model.CampaignExperiment;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandLiftExperimentsService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.ExperimentRetargetingConditions;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.feature.FeatureName.BRAND_LIFT_HIDDEN;
import static ru.yandex.direct.feature.FeatureName.CPM_GLOBAL_AB_SEGMENT;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBrandLiftAddOperationSupport extends AbstractCampaignAddOperationSupport<CampaignWithBrandLift> {

    private final RetargetingConditionService retargetingConditionService;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;
    private final CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService;

    @Autowired
    public CampaignWithBrandLiftAddOperationSupport(RetargetingConditionService retargetingConditionService,
                                                    ShardHelper shardHelper,
                                                    FeatureService featureService,
                                                    CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService) {
        this.retargetingConditionService = retargetingConditionService;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.campaignWithBrandLiftExperimentsService = campaignWithBrandLiftExperimentsService;
    }

    @Override
    public Class<CampaignWithBrandLift> getTypeClass() {
        return CampaignWithBrandLift.class;
    }

    @Override
    public void onPreValidated(RestrictedCampaignsAddOperationContainer addContainer,
                               List<CampaignWithBrandLift> models) {
        boolean brandLiftHiddenEnabledForOperator = featureService.isEnabled(addContainer.getOperatorUid(),
                BRAND_LIFT_HIDDEN);
        boolean globalABSegment = featureService.isEnabledForClientId(addContainer.getClientId(),
                CPM_GLOBAL_AB_SEGMENT);

        models.forEach(model ->
                model.setIsBrandLiftHidden(model.getBrandSurveyId() != null && brandLiftHiddenEnabledForOperator));
        models.forEach(model -> model.setIsCpmGlobalAbSegment(globalABSegment && model.getBrandSurveyId() == null && !nvl(model.getIsSearchLiftEnabled(), false)));
    }

    @Override
    public void beforeExecution(RestrictedCampaignsAddOperationContainer addContainer,
                                List<CampaignWithBrandLift> models) {
        // При копировании брендлифтовых кампаний у которых brandSurveyId == null, в препроцессорах копирования не будут
        // сброшены поля abSegmentRetargetingConditionId, abSegmentStatisticRetargetingConditionId, sectionIds и
        // abSegmentGoalIds. Весь код ниже пытается по abSegmentGoalIds и sectionIds найти валидные значения полей
        // abSegmentRetargetingConditionId и abSegmentStatisticRetargetingConditionId в методе
        // retargetingConditionService.findOrCreateExperimentsRetargetingConditions(), но если метрика вернет
        // пустой список целей для клиента в addContainer.getMetrikaClient().getAbSegmentGoals(), то код ниже упадет.
        // При копировании мы не хотим изменять поля abSegmentRetargetingConditionId и
        // abSegmentStatisticRetargetingConditionId и проверять наличие целей нам не нужно, так же поступает
        // копирование в перле. Поэтому выполнять весь код ниже при копировании смысла нет, по крайней мере
        // до того момента, когда мы решим, что валидация целей экспериментов при копировании нам необходима.
        if (addContainer.isCopy()) {
            return;
        }

        String clientLogin = shardHelper.getLoginByUid(addContainer.getChiefUid());

        var brandLifts =
                campaignWithBrandLiftExperimentsService.clientBrandLifts(addContainer.getClientId()).stream()
                        .collect(Collectors.toMap(BrandSurvey::getBrandSurveyId, Function.identity()));

        var modelsWithoutBrandLifts = new ArrayList<CampaignWithBrandLift>();
        var modelsWithExistedBrandLifts = new ArrayList<CampaignWithBrandLift>();
        var modelsWithoutExistedBrandLifts = new ArrayList<CampaignWithBrandLift>();

        for (var campaign : models) {
            if (campaign.getBrandSurveyId() == null) {
                modelsWithoutBrandLifts.add(campaign);
            } else {
                if (brandLifts.containsKey(campaign.getBrandSurveyId())) {
                    modelsWithExistedBrandLifts.add(campaign);
                } else {
                    modelsWithoutExistedBrandLifts.add(campaign);
                }
            }
        }

        // Создаем эксперимент в случае, если нет лифтов и включена фича на единый контроль или есть только серч лифт
        var preparedExperimentsGlobal = StreamEx.of(modelsWithoutBrandLifts)
                    .filter(model -> nvl(model.getIsCpmGlobalAbSegment(), false) || nvl(model.getIsSearchLiftEnabled(), false))
                    .flatMapToEntry(model -> campaignWithBrandLiftExperimentsService.prepareExperiment(clientLogin, model))
                    .grouping();
        var experimentRetargetingConditions1 = getRetargetingConditions(modelsWithoutBrandLifts, addContainer, preparedExperimentsGlobal);
        EntryStream.of(modelsWithoutBrandLifts)
                .forKeyValue((index, ac) -> {
                    var retargetingConditions = experimentRetargetingConditions1.get(index);
                    updateCampaignWithExperimentRetargetingConditions(ac, retargetingConditions);
                });

        // Создаем эксперимент и заполняем табличку brandSurvey в случае,
        // был присоединен новый BL
        var preparedExperiments = StreamEx.of(modelsWithoutExistedBrandLifts)
                .flatMapToEntry(model -> campaignWithBrandLiftExperimentsService.prepareExperiment(clientLogin, model))
                .grouping();
        var experimentRetargetingConditions = getRetargetingConditions(modelsWithoutExistedBrandLifts, addContainer, preparedExperiments);
        EntryStream.of(modelsWithoutExistedBrandLifts)
                .forKeyValue((index, ac) -> {
                    var retargetingConditions = experimentRetargetingConditions.get(index);
                    updateCampaignWithExperimentRetargetingConditions(ac, retargetingConditions);
                    if (ac.getBrandSurveyId() != null) {
                        String brandSurveyName = nvl(ac.getBrandSurveyName(), "").isBlank() ?
                                defaultBrandLiftName(clientLogin) :
                                ac.getBrandSurveyName();

                        var brandSurvey = new BrandSurvey()
                                .withClientId(addContainer.getClientId().asLong())
                                .withBrandSurveyId(ac.getBrandSurveyId())
                                .withName(brandSurveyName)
                                .withRetargetingConditionId(retargetingConditions
                                        .getRetargetingConditionId())
                                .withIsBrandLiftHidden(ac.getIsBrandLiftHidden())
                                .withExperimentId(ac.getExperimentId())
                                .withSegmentId(Optional.ofNullable(preparedExperiments
                                        .get(ac.getExperimentId()))
                                        .map(list -> list.get(0))
                                        .orElse(null));

                        campaignWithBrandLiftExperimentsService
                                .addBrandSurvey(clientLogin, brandSurvey);
                    }
                });

        // В случае, если присоединяем существующий BL,
        // берем из него эксперимент с сегментом и если надо, переименовываем BL
        Map<Long, List<Long>> preparedExperimentsWithExistedBl = StreamEx.of(modelsWithExistedBrandLifts)
                .flatMapToEntry(ac -> {
                    var bl = brandLifts.get(ac.getBrandSurveyId());
                    ac.setSectionIds(nvl(ac.getSectionIds(), new ArrayList<>()));
                    ac.setAbSegmentGoalIds(nvl(ac.getAbSegmentGoalIds(), new ArrayList<>()));

                    ac.getSectionIds().add(nvl(bl.getExperimentId(), 0L));
                    ac.getAbSegmentGoalIds().add(nvl(bl.getSegmentId(), 0L));
                    return Map.of(nvl(bl.getExperimentId(), 0L), nvl(bl.getSegmentId(), 0L));
                }).grouping();
        var experimentRetargetingConditionsWithExistedBl = getRetargetingConditions(modelsWithExistedBrandLifts, addContainer, preparedExperimentsWithExistedBl);
        EntryStream.of(modelsWithExistedBrandLifts)
                .forKeyValue((index, ac) -> {
                    var brandSurvey = brandLifts.get(ac.getBrandSurveyId());
                    updateCampaignWithExperimentRetargetingConditions(ac,
                            experimentRetargetingConditionsWithExistedBl.get(index));
                    if (ac.getBrandSurveyName() != null &&
                            !brandSurvey.getName().equals(ac.getBrandSurveyName())) {
                        campaignWithBrandLiftExperimentsService
                                .renameBrandSurvey(clientLogin, brandSurvey.getBrandSurveyId(), ac.getBrandSurveyName());
                    }
                });
    }

    private List<ExperimentRetargetingConditions> getRetargetingConditions(List<CampaignWithBrandLift> campaigns,
                                                                           RestrictedCampaignsAddOperationContainer addContainer,
                                                                           Map<Long, List<Long>> preparedExperiments) {
        if (CollectionUtils.isEmpty(campaigns)) {
            return null;
        }
        List<CampaignExperiment> campaignExperiments = mapList(campaigns,
                item -> new CampaignExperiment()
                        .withAbSegmentGoalIds(item.getAbSegmentGoalIds())
                        .withSectionIds(item.getSectionIds()));
        return retargetingConditionService.findOrCreateExperimentsRetargetingConditions(
                        addContainer.getClientId(),
                        campaignExperiments,
                        addContainer.getMetrikaClient().getAbSegmentGoals(),
                        preparedExperiments);
    }

    private void updateCampaignWithExperimentRetargetingConditions(
            CampaignWithBrandLift campaign,
            ExperimentRetargetingConditions experimentRetargetingConditions) {

        campaign.setAbSegmentRetargetingConditionId(experimentRetargetingConditions
                .getRetargetingConditionId());

        campaign.setAbSegmentStatisticRetargetingConditionId(experimentRetargetingConditions
                .getStatisticRetargetingConditionId());
    }

    private void updateCampaignWithBrandLiftRetargetingConditions(
            CampaignWithBrandLift campaign, long retargetingConditionId) {

        campaign.setAbSegmentRetargetingConditionId(retargetingConditionId);

        campaign.setAbSegmentStatisticRetargetingConditionId(retargetingConditionId);
    }

    /**
     * При созданиии эксперимента в Аудитория в beforeExecution у нас нет campaignId, он проставляется в execute.
     * Поэтому в afterExecution переименовываем эксперимент в "Brand-lift $campaignId"
     */
    @Override
    public void afterExecution(RestrictedCampaignsAddOperationContainer addContainer,
                               List<CampaignWithBrandLift> models) {
        String clientLogin = shardHelper.getLoginByUid(addContainer.getChiefUid());
        EntryStream.of(models).forKeyValue((index, model) -> {

            //experimentId проставляется только при создании BL, при переиспользовании null
            if (model.getBrandSurveyId() != null && model.getExperimentId() != null) {
                checkNotNull(clientLogin,
                        "login for creating brandlift is null for ClientId=" + addContainer.getClientId());
                campaignWithBrandLiftExperimentsService.renameExperiment(clientLogin, model);
                if (nvl(model.getBrandSurveyName(), "").isEmpty()) {
                    campaignWithBrandLiftExperimentsService
                            .renameBrandSurvey(clientLogin, model.getBrandSurveyId(), "Brand-lift " + model.getId());
                }
            }
        });

    }

    private String defaultBrandLiftName(String clientLogin) {
        return "Brand-lift for " + clientLogin;
    }
}
