package ru.yandex.direct.core.entity.campaign.service.type.add;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyService;
import ru.yandex.direct.core.entity.campaign.service.pricerecalculation.CommonCampaignPriceRecalculationService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.feature.FeatureName;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithCustomStrategyAddOperationSupport extends AbstractCampaignAddOperationSupport<CampaignWithCustomStrategy> {
    private final CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService;
    private final FeatureService featureService;

    @Autowired
    public CampaignWithCustomStrategyAddOperationSupport(
            CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService,
            FeatureService featureService) {
        this.commonCampaignPriceRecalculationService = commonCampaignPriceRecalculationService;
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithCustomStrategy> getTypeClass() {
        return CampaignWithCustomStrategy.class;
    }

    @Override
    public void onModelsValidated(RestrictedCampaignsAddOperationContainer addModelContainer,
                                  List<CampaignWithCustomStrategy> models) {
        boolean conversionStrategyLearningStatusEnabled =
                featureService.isEnabledForClientId(addModelContainer.getClientId(),
                        FeatureName.CONVERSION_STRATEGY_LEARNING_STATUS_ENABLED);

        if (conversionStrategyLearningStatusEnabled) {
            LocalDateTime now = now();
            StreamEx.of(models)
                    .map(CampaignWithCustomStrategy::getStrategy)
                    .map(DbStrategyBase::getStrategyData)
                    .filter(CampaignWithStrategyValidationUtils::isStrategyWithSupportOfLearningStatus)
                    .forEach(data -> data.setLastBidderRestartTime(now));
        }
    }

    @Override
    public void beforeExecution(RestrictedCampaignsAddOperationContainer addContainer,
                                List<CampaignWithCustomStrategy> models) {
        if (featureService.isEnabledForClientId(addContainer.getClientId(), FeatureName.CPV_PAY_FOR_TRUE_VIEW)) {
            models.forEach(model -> {
                DbStrategy newStrategy = CampaignStrategyService
                        .correctCampaignStrategyUsePayForConversionValue(model.getStrategy());
                model.withStrategy(newStrategy);
                }
            );
        }
    }

    @Override
    public void afterExecution(RestrictedCampaignsAddOperationContainer addContainer,
                               List<CampaignWithCustomStrategy> campaigns) {

        List<Long> campaignWithAutobudgetStrategyIds = filterAndMapList(campaigns,
                campaign -> campaign.getStrategy().isAutoBudget(),
                CampaignWithCustomStrategy::getId);
        commonCampaignPriceRecalculationService
                .addAutobudgetForecast(addContainer.getShard(), campaignWithAutobudgetStrategyIds);
    }

}
