package ru.yandex.direct.core.entity.campaign.service.type.add;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.WithMetrikaCounters;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainerService;
import ru.yandex.direct.core.entity.strategy.container.StrategyOperationOptions;
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainerService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCampaignIds;
import ru.yandex.direct.core.entity.strategy.service.add.StrategyAddOperationService;
import ru.yandex.direct.core.entity.strategy.service.converter.CampaignToStrategyConverterService;
import ru.yandex.direct.core.entity.strategy.service.converter.StrategyToCampaignConverterFacade;
import ru.yandex.direct.core.entity.strategy.service.update.StrategyUpdateOperationService;
import ru.yandex.direct.core.entity.strategy.type.common.CommonStrategyUpdateLastChangeService;
import ru.yandex.direct.model.ModelChanges;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.common.db.PpcPropertyNames.MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.feature.FeatureName.GET_STRATEGY_ID_FROM_SHARD_INC_STRATEGY_ID;
import static ru.yandex.direct.feature.FeatureName.PACKAGE_STRATEGIES_STAGE_TWO;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

//todo: включить компонент, как будут готовы репозиторный и сервисный уровни стратегий
@Component
@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyAddOperationSupport
        extends AbstractCampaignAddOperationSupport<CampaignWithPackageStrategy> {

    private final CampaignToStrategyConverterService campaignToStrategyConverterService;
    private final StrategyToCampaignConverterFacade strategyToCampaignConverterFacade;
    private final StrategyAddOperationService strategyAddOperationService;
    private final StrategyUpdateOperationService strategyUpdateOperationService;
    private final FeatureService featureService;
    private final PpcProperty<Integer> maxCampaignsAllowedToLinkToStrategyProperty;
    private final StrategyAddOperationContainerService strategyAddOperationContainerService;
    private final CommonStrategyUpdateLastChangeService commonStrategyUpdateLastChangeService;
    private final StrategyUpdateOperationContainerService strategyUpdateOperationContainerService;

    @Autowired
    public CampaignWithPackageStrategyAddOperationSupport(
            CampaignToStrategyConverterService campaignToStrategyConverterService,
            StrategyToCampaignConverterFacade strategyToCampaignConverterFacade,
            StrategyAddOperationService strategyAddOperationService,
            StrategyUpdateOperationService strategyUpdateOperationService,
            FeatureService featureService,
            PpcPropertiesSupport ppcPropertiesSupport,
            StrategyAddOperationContainerService strategyAddOperationContainerService,
            CommonStrategyUpdateLastChangeService commonStrategyUpdateLastChangeService,
            StrategyUpdateOperationContainerService strategyUpdateOperationContainerService) {
        this.campaignToStrategyConverterService = campaignToStrategyConverterService;
        this.strategyToCampaignConverterFacade = strategyToCampaignConverterFacade;
        this.strategyAddOperationService = strategyAddOperationService;
        this.strategyUpdateOperationService = strategyUpdateOperationService;
        this.featureService = featureService;
        this.maxCampaignsAllowedToLinkToStrategyProperty =
                ppcPropertiesSupport.get(MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY);
        this.strategyAddOperationContainerService = strategyAddOperationContainerService;
        this.commonStrategyUpdateLastChangeService = commonStrategyUpdateLastChangeService;
        this.strategyUpdateOperationContainerService = strategyUpdateOperationContainerService;
    }

    @Override
    public void onPreValidated(RestrictedCampaignsAddOperationContainer addContainer,
                               List<CampaignWithPackageStrategy> campaigns) {
        boolean isPackageStrategiesStageTwoFeatureEnabled =
                featureService.isEnabledForClientId(addContainer.getClientId(),
                        PACKAGE_STRATEGIES_STAGE_TWO);
        if (isPackageStrategiesStageTwoFeatureEnabled) {
            LocalDateTime now = now();
            StreamEx.of(campaigns)
                    .remove(x -> x.getStrategyId() == null)
                    .mapToEntry(CampaignWithPackageStrategy::getStrategyId)
                    .mapValues(addContainer::getPackageStrategy)
                    .forKeyValue((campaign, strategy) -> copyStrategyToCampaign(now, campaign, strategy));
        } else {
            campaigns.forEach(c -> c.setStrategyId(0L));
        }

        StreamEx.of(campaigns)
                .filter(c -> c.getStrategyId() == null)
                .forEach(c -> c.setStrategyId(0L)); // позже обновим из операции стратегии на верный
        Set<Long> newMetrikaCounterIds = StreamEx.of(campaigns)
                .select(WithMetrikaCounters.class)
                .map(WithMetrikaCounters::getMetrikaCounters)
                .nonNull()
                .flatMap(Collection::stream)
                .toSet();
        addContainer.getMetrikaClient().setCampaignsCounterIds(campaigns, newMetrikaCounterIds);
    }

    private void copyStrategyToCampaign(LocalDateTime now, CampaignWithPackageStrategy campaign,
                                        BaseStrategy baseStrategy) {
        CampaignsPlatform platform = campaign.getStrategy().getPlatform();
        CampOptionsStrategy options = campaign.getStrategy().getStrategy();

        strategyToCampaignConverterFacade.copyStrategyToCampaign(now, baseStrategy, campaign);
        campaign.getStrategy().setPlatform(platform);
        campaign.getStrategy().setStrategy(options);
    }

    @Override
    public void updateRelatedEntitiesInTransaction(
            DSLContext dslContext,
            RestrictedCampaignsAddOperationContainer addModelContainer,
            List<CampaignWithPackageStrategy> campaigns) {
        boolean isPackageStrategiesStageTwoFeatureEnabled =
                featureService.isEnabledForClientId(addModelContainer.getClientId(),
                        PACKAGE_STRATEGIES_STAGE_TWO);
        boolean isGetStrategyIdFromShardIncStrategyIdEnabled =
                featureService.isEnabledForClientId(addModelContainer.getClientId(),
                        GET_STRATEGY_ID_FROM_SHARD_INC_STRATEGY_ID);

        LocalDateTime now = now();
        var campaignsWithExistingStrategies = filterList(
                campaigns, campaign -> campaign.getStrategyId() != null && campaign.getStrategyId() != 0
        );

        if (isPackageStrategiesStageTwoFeatureEnabled) {
            updateStrategiesThatBecomePublic(dslContext, addModelContainer, campaigns, now);
        }

        //для кампаний, у которых не указан strategy_id, создаем пакетную стратегию, на первом этапе это все
        // стратегии
        List<CampaignWithPackageStrategy> campaignsToCreatePackage = filterList(campaigns,
                campaign -> campaign.getStrategyId() == 0);

        var strategiesToAddByCampaignId = listToMap(
                campaignsToCreatePackage,
                BaseCampaign::getId,
                campaign ->
                        isGetStrategyIdFromShardIncStrategyIdEnabled
                                ? campaignToStrategyConverterService.toStrategyWithNotFilledId(
                                addModelContainer.getClientId(), now, campaign)
                                : campaignToStrategyConverterService.toStrategyWithIdEqualToCidWithOffset(
                                addModelContainer.getClientId(), now, campaign));
        StrategyOperationOptions options = new StrategyOperationOptions(
                false,
                false,
                false,
                false,
                !isGetStrategyIdFromShardIncStrategyIdEnabled,
                false,
                true,
                maxCampaignsAllowedToLinkToStrategyProperty.getOrDefault(DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY));
        StrategyAddOperationContainer container = new StrategyAddOperationContainer(addModelContainer.getShard(),
                addModelContainer.getClientId(), addModelContainer.getOperatorUid(),
                addModelContainer.getClientUid(), options);
        var strategiesToAdd = List.copyOf(strategiesToAddByCampaignId.values());
        strategyAddOperationContainerService.fillContainers(container, strategiesToAdd, dslContext);
        //линковка стратегий к кампаниям происходит внутри strategyAddOperationService.execute
        strategyAddOperationService.execute(dslContext, container, strategiesToAdd);
        //заполняем id стратегий внутри кампаний
        campaignsToCreatePackage.forEach(campaign -> campaign.setStrategyId(strategiesToAddByCampaignId.get(campaign.getId()).getId()));
        // обновляем уже существующие стратегии
        commonStrategyUpdateLastChangeService.updateLastChangeForCampaignsStrategies(
                dslContext, container, campaignsWithExistingStrategies
        );
    }

    private void updateStrategiesThatBecomePublic(DSLContext dslContext,
                                                  RestrictedCampaignsAddOperationContainer addModelContainer,
                                                  List<CampaignWithPackageStrategy> campaigns,
                                                  LocalDateTime now) {
        Map<Long, CommonStrategy> strategiesToBecomePublicById = StreamEx.of(campaigns)
                .map(CampaignWithPackageStrategy::getStrategyId)
                .filter(strategyId -> strategyId != 0)
                .distinct()
                .mapToEntry(strategyId -> (CommonStrategy) addModelContainer.getPackageStrategy(strategyId))
                .filterValues(strategy -> shouldBecomePublic(strategy, campaigns))
                .toMap();

        List<ModelChanges<CommonStrategy>> strategiesToBecomePublicModelChanges = mapList(
                strategiesToBecomePublicById.values(),
                s -> new ModelChanges<>(s.getId(), CommonStrategy.class)
                        .process(true, CommonStrategy.IS_PUBLIC)
                        .process(String.format("Strategy %d dated %s", s.getId(), now.toLocalDate()),
                                CommonStrategy.NAME));

        strategyUpdateOperationService.executeFromCampaign(dslContext, addModelContainer, strategiesToBecomePublicById,
                strategiesToBecomePublicModelChanges);
    }

    private static boolean shouldBecomePublic(BaseStrategy strategy, List<CampaignWithPackageStrategy> campaigns) {
        var isPublic = ((CommonStrategy) strategy).getIsPublic();
        var isNotPublicStrategy = isPublic == null || !isPublic;

        List<Long> cids = ((StrategyWithCampaignIds) strategy).getCids();
        Set<Long> setOfOldCids = cids == null ? Collections.emptySet() : Set.copyOf(cids);
        List<Long> newCampaignsToLink = mapList(
                filterList(
                        campaigns,
                        c -> strategy.getId().equals(c.getStrategyId())
                ),
                BaseCampaign::getId);
        var willHaveNewLinkedCampaigns = !setOfOldCids.containsAll(newCampaignsToLink);

        return isNotPublicStrategy && willHaveNewLinkedCampaigns;
    }

    @Override
    public Class<CampaignWithPackageStrategy> getTypeClass() {
        return CampaignWithPackageStrategy.class;
    }
}
