package ru.yandex.direct.core.entity.campaign.service.type.add;

import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusCorrect;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.utils.CollectionUtils;

import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.enrichBidModifierInventory;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.extractImpressionRateIntervalDaysFromPackage;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.getStrategy;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.pricePackageHasImpressionRate;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPricePackageAddOperationSupport
        extends AbstractCampaignAddOperationSupport<CampaignWithPricePackage> {
    private final PricePackageRepository pricePackageRepository;
    private final ClientService clientService;
    private final ProductService productService;
    private final FeatureService featureService;

    @Autowired
    public CampaignWithPricePackageAddOperationSupport(
            PricePackageRepository pricePackageRepository,
            ClientService clientService,
            ProductService productService,
            FeatureService featureService) {
        this.pricePackageRepository = pricePackageRepository;
        this.clientService = clientService;
        this.productService = productService;
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithPricePackage> getTypeClass() {
        return CampaignWithPricePackage.class;
    }

    @Override
    public void onPreValidated(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                               List<CampaignWithPricePackage> campaigns) {
        List<CampaignWithPricePackage> campaignsWithPackageId = filterList(campaigns,
                x -> x.getPricePackageId() != null);
        var pricePackageIds = listToSet(campaignsWithPackageId, CampaignWithPricePackage::getPricePackageId);
        var pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);
        var calcBudgetFeature = featureService.isEnabledForClientId(addCampaignParametersContainer.getClientId(),
                FeatureName.BACKEND_CPM_PRICE_CAMPAIGN_BUDGET_CALC_ENABLED);

        for (var campaign : campaignsWithPackageId) {
            var pricePackage = pricePackages.get(campaign.getPricePackageId());

            campaign
                    .withProductId(calculateProductId(addCampaignParametersContainer.getClientId(), pricePackage))
                    .withFlightStatusApprove(newCampStatusApprove(pricePackage))
                    .withIsDraftApproveAllowed(pricePackage.getIsDraftApproveAllowed())
                    .withFlightStatusCorrect(PriceFlightStatusCorrect.NEW)
                    .withStrategy(getStrategy(campaign, pricePackage, calcBudgetFeature));
            if (!pricePackage.isFrontpagePackage() && !pricePackage.isFrontpageVideoPackage()) {
                // опустошаем список площадок, даже если он пришёл с фронта непустой
                // прайсовое видео не может показываться на этих площадках
                campaign.getFlightTargetingsSnapshot().withViewTypes(Collections.emptyList());
            }
            if (pricePackage != null && !CollectionUtils.isEmpty(pricePackage.getAllowedPageIds())) {
                campaign.setAllowedPageIds(pricePackage.getAllowedPageIds());
            }
            if (pricePackageHasImpressionRate(pricePackage)
                    && campaign.getImpressionRateCount() == null && campaign.getImpressionRateIntervalDays() == null) {
                //Если в пакете явно указана частота и с фронта она не пришла явно, она копируется на РК с пакета.
                campaign.setImpressionRateCount(
                        pricePackage.getCampaignOptions().getShowsFrequencyLimit().getFrequencyLimit()
                );
                campaign.setImpressionRateIntervalDays(extractImpressionRateIntervalDaysFromPackage(pricePackage));
            }
            campaign.setBidModifiers(enrichBidModifierInventory(pricePackage, campaign));
        }
    }

    private Long calculateProductId(ClientId clientId, PricePackage pricePackage) {
        Client client = clientService.getClient(clientId);
        return productService.calculateProductIdByPackage(client, pricePackage);
    }

    private static PriceFlightStatusApprove newCampStatusApprove(PricePackage pricePackage) {
        return pricePackage != null && pricePackage.getCampaignAutoApprove() != null
                && pricePackage.getCampaignAutoApprove() ?
                PriceFlightStatusApprove.YES :
                PriceFlightStatusApprove.NEW;
    }
}
