package ru.yandex.direct.core.entity.campaign.service.type.add;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueObjType;
import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueuePriority;
import ru.yandex.direct.core.entity.balance.model.BalanceNotificationInfo;
import ru.yandex.direct.core.entity.balance.service.BalanceInfoQueueService;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSendingToBalance;
import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignBalanceService;
import ru.yandex.direct.core.entity.campaign.service.CommonCampaignService;
import ru.yandex.direct.core.entity.campaign.service.WalletService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.AddServicedCampaignInfo;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.campaign.service.type.add.utils.CampaignAddTypeSupportUtilsKt.getManagerUidWithServicedCampaignIds;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithSendingToBalanceAddOperationSupport
        extends AbstractCampaignAddOperationSupport<CampaignWithSendingToBalance> {

    private final CampaignRepository campaignRepository;
    private final CampaignBalanceService campaignBalanceService;
    private final AddServicedCampaignService addServicedCampaignService;
    private final ClientService clientService;
    private final WalletService walletService;
    private final CommonCampaignService commonCampaignService;
    private final UserService userService;
    private final BalanceInfoQueueService balanceInfoQueueService;
    private final DslContextProvider ppcDslContextProvider;

    @Autowired
    public CampaignWithSendingToBalanceAddOperationSupport(CampaignRepository campaignRepository,
                                                           CampaignBalanceService campaignBalanceService,
                                                           AddServicedCampaignService addServicedCampaignService,
                                                           ClientService clientService, WalletService walletService,
                                                           CommonCampaignService commonCampaignService,
                                                           UserService userService,
                                                           BalanceInfoQueueService balanceInfoQueueService,
                                                           DslContextProvider ppcDslContextProvider) {
        this.campaignRepository = campaignRepository;
        this.campaignBalanceService = campaignBalanceService;
        this.addServicedCampaignService = addServicedCampaignService;
        this.clientService = clientService;
        this.walletService = walletService;
        this.commonCampaignService = commonCampaignService;
        this.userService = userService;
        this.balanceInfoQueueService = balanceInfoQueueService;
        this.ppcDslContextProvider = ppcDslContextProvider;
    }

    @Override
    public Class<CampaignWithSendingToBalance> getTypeClass() {
        return CampaignWithSendingToBalance.class;
    }

    @Override
    public void afterExecution(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                               List<CampaignWithSendingToBalance> campaigns) {
        Client client = clientService.getClient(addCampaignParametersContainer.getClientId());
        List<AddServicedCampaignInfo> servicedInfos =
                addServicedCampaignService.getServicedInfoForNewCampaigns(addCampaignParametersContainer, campaigns);

        var managerUidWithServicedCampaignIds = getManagerUidWithServicedCampaignIds(servicedInfos, campaigns);
        Long managerUid = Optional.ofNullable(managerUidWithServicedCampaignIds)
                .map(UidWithCampaignIds::getUid)
                .orElse(null);
        var servicedCampaignIds = Optional.ofNullable(managerUidWithServicedCampaignIds)
                .map(UidWithCampaignIds::getCampaignIds)
                .orElse(emptyList());

        @Nullable UidAndClientId agencyUidAndClientId = userService.getAgencyUidAndClientId(
                addCampaignParametersContainer.getOperatorUid(), client.getAgencyUserId(), client.getAgencyClientId());

        List<BalanceNotificationInfo> balanceNotificationInfosOnServiced = emptyList();
        if (managerUid != null) {
            User manager = userService.getUser(managerUid);
            checkState(manager != null, "not found manager with uid %d", managerUid);
            balanceNotificationInfosOnServiced = onCampaignManagerChanged(servicedCampaignIds, manager);
        }

        boolean isFirstCampaignsUnderWallet =
                campaignRepository.isFirstCampaignsUnderWallet(addCampaignParametersContainer.getShard(),
                        addCampaignParametersContainer.getClientId(),
                        mapList(campaigns, CampaignWithSendingToBalance::getType),
                        listToSet(campaigns, CampaignWithSendingToBalance::getId));

        List<BalanceNotificationInfo> balanceNotificationInfosOnBalanceUpdateOrderFail = emptyList();

        if (isFirstCampaignsUnderWallet && !client.getSharedAccountDisabled()) {
            balanceNotificationInfosOnBalanceUpdateOrderFail = onFirstNewCampaigns(addCampaignParametersContainer,
                    client, campaigns, managerUidWithServicedCampaignIds, agencyUidAndClientId);
        }

        List<CampaignWithSendingToBalance> campaignsNotUnderWallet =
                filterList(campaigns, c -> !CampaignTypeKinds.UNDER_WALLET.contains(c.getType()));
        if (!campaignsNotUnderWallet.isEmpty()) {
            campaignBalanceService.createOrUpdateOrdersOnFirstCampaignCreation(
                    addCampaignParametersContainer.getOperatorUid(),
                    agencyUidAndClientId,
                    managerUidWithServicedCampaignIds,
                    campaignsNotUnderWallet);
        }

        DSLContext context = ppcDslContextProvider.ppc(addCampaignParametersContainer.getShard());
        List<BalanceNotificationInfo> balanceNotificationInfos = StreamEx.of(campaigns)
                .map(c -> getNotificationInfoOnCreatingCamp(addCampaignParametersContainer.getClientUid(),
                        addCampaignParametersContainer.getOperatorUid()))
                .append(balanceNotificationInfosOnServiced)
                .append(balanceNotificationInfosOnBalanceUpdateOrderFail)
                .toList();

        balanceInfoQueueService.addToBalanceInfoQueue(context, balanceNotificationInfos);
    }

    private List<BalanceNotificationInfo> onFirstNewCampaigns(
            RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
            Client client,
            List<CampaignWithSendingToBalance> campaigns,
            UidWithCampaignIds managerUidWithServicedCampaignIds,
            UidAndClientId agencyUidAndClientId) {

        if (client.getWorkCurrency() != CurrencyCode.YND_FIXED) {
            Wallet wallet = walletService.getWalletForNewCampaigns(addCampaignParametersContainer,
                    agencyUidAndClientId);

            if (wallet != null) {
                commonCampaignService.turnOnWalletStatusAggregated(addCampaignParametersContainer,
                        wallet.getWalletCampaignId());
            }
        }

        List<Long> walletIds = mapList(campaigns, CampaignWithSendingToBalance::getWalletId);
        List<WalletTypedCampaign> wallets = commonCampaignService.getWalletsByIds(
                addCampaignParametersContainer.getShard(), walletIds);

        boolean creatingOrderInBalanceForWalletsIsSuccessful =
                campaignBalanceService.createOrUpdateOrdersOnFirstCampaignCreation(
                        addCampaignParametersContainer.getOperatorUid(),
                        agencyUidAndClientId,
                        managerUidWithServicedCampaignIds,
                        wallets);

        boolean creatingOrderInBalanceForCampaignsIsSuccessful =
                campaignBalanceService.createOrUpdateOrdersOnFirstCampaignCreation(
                        addCampaignParametersContainer.getOperatorUid(),
                        agencyUidAndClientId,
                        managerUidWithServicedCampaignIds,
                        campaigns);

        if (!creatingOrderInBalanceForWalletsIsSuccessful || !creatingOrderInBalanceForCampaignsIsSuccessful) {
            return mapList(campaigns, c -> getNotificationInfoOnBalanceUpdateOrderFail(c.getId(),
                    addCampaignParametersContainer.getOperatorUid()));
        }

        return emptyList();
    }

    private List<BalanceNotificationInfo> onCampaignManagerChanged(List<Long> campaignIds,
                                                                   User manager) {
        return mapList(campaignIds, campaignId -> getNotificationInfoOnServicedCamp(campaignId, manager.getUid()));
    }

    private static BalanceNotificationInfo getNotificationInfoOnCreatingCamp(Long subjectUserUid, Long operatorUid) {
        return new BalanceNotificationInfo()
                .withCidOrUid(subjectUserUid)
                .withObjType(BalanceInfoQueueObjType.UID)
                .withPriority(BalanceInfoQueuePriority.USER_ON_SAVING_NEW_CAMPAIGN)
                .withOperatorUid(operatorUid);
    }

    private static BalanceNotificationInfo getNotificationInfoOnServicedCamp(Long cid, Long managerUid) {
        return new BalanceNotificationInfo()
                .withCidOrUid(cid)
                .withObjType(BalanceInfoQueueObjType.CID)
                .withPriority(BalanceInfoQueuePriority.PRIORITY_CAMP_ON_MANAGER_CHANGED)
                .withOperatorUid(managerUid);
    }

    private static BalanceNotificationInfo getNotificationInfoOnBalanceUpdateOrderFail(Long cid, Long operatorUid) {
        return new BalanceNotificationInfo()
                .withCidOrUid(cid)
                .withObjType(BalanceInfoQueueObjType.CID)
                .withPriority(BalanceInfoQueuePriority.PRIORITY_CAMPS_ON_ENABLE_WALLET)
                .withOperatorUid(operatorUid);
    }

}
