package ru.yandex.direct.core.entity.campaign.service.type.add;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.BillingAggregateService;
import ru.yandex.direct.core.entity.campaign.service.WalletService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.AddServicedCampaignInfo;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.AutoServicingMailNotification;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.UserPerminfo;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.util.RepositoryUtils.NOW_PLACEHOLDER;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CommonCampaignAddOperationSupport extends AbstractCampaignAddOperationSupport<CommonCampaign> {
    private final CampaignRepository campaignRepository;

    private final AddServicedCampaignService addServicedCampaignService;
    private final ClientService clientService;
    private final WalletService walletService;
    private final BillingAggregateService billingAggregateService;
    private final ProductService productService;
    private final UserService userService;
    private final PpcRbac ppcRbac;
    private final FeatureService featureService;
    private final NotificationService notificationService;

    @Autowired
    public CommonCampaignAddOperationSupport(
            CampaignRepository campaignRepository,
            AddServicedCampaignService addServicedCampaignService,
            ClientService clientService,
            WalletService walletService,
            BillingAggregateService billingAggregateService,
            ProductService productService,
            UserService userService,
            PpcRbac ppcRbac,
            FeatureService featureService,
            NotificationService notificationService) {

        this.campaignRepository = campaignRepository;
        this.addServicedCampaignService = addServicedCampaignService;
        this.clientService = clientService;
        this.walletService = walletService;
        this.billingAggregateService = billingAggregateService;
        this.productService = productService;
        this.userService = userService;
        this.ppcRbac = ppcRbac;
        this.featureService = featureService;
        this.notificationService = notificationService;
    }

    @Override
    public Class<CommonCampaign> getTypeClass() {
        return CommonCampaign.class;
    }

    @Override
    public void onPreValidated(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                               List<CommonCampaign> campaigns) {
        Client client = clientService.getClient(addCampaignParametersContainer.getClientId());
        checkState(client != null, "Invalid clientId: " + addCampaignParametersContainer.getClientId());

        List<AddServicedCampaignInfo> servicedInfos =
                addServicedCampaignService.getServicedInfoForNewCampaigns(addCampaignParametersContainer, campaigns);
        @Nullable UidAndClientId agencyUidAndClientId = userService.getAgencyUidAndClientId(
                addCampaignParametersContainer.getOperatorUid(), client.getAgencyUserId(), client.getAgencyClientId());

        Wallet wallet = walletService.getWalletForNewCampaigns(addCampaignParametersContainer, agencyUidAndClientId);

        User subjectUser = userService.getUser(addCampaignParametersContainer.getClientUid());

        boolean isFirstCampaignsUnderWallet =
                campaignRepository.isFirstCampaignsUnderWallet(addCampaignParametersContainer.getShard(),
                        addCampaignParametersContainer.getClientId(),
                        mapList(campaigns, CommonCampaign::getType),
                        Collections.emptySet());

        for (int i = 0; i < campaigns.size(); i++) {
            var campaign = campaigns.get(i);
            if (campaign.getFio() == null) {
                campaign.setFio(subjectUser.getFio());
            }

            setZeroSumValues(campaign);

            setProductId(client, campaign);
            if (wallet != null && (wallet.getEnabled() || isFirstCampaignsUnderWallet)) {
                campaign.setWalletId(wallet.getWalletCampaignId());
            } else {
                campaign.setWalletId(0L);
            }

            campaign.setUid(addCampaignParametersContainer.getChiefUid());
            campaign.setClientId(addCampaignParametersContainer.getClientId().asLong());
            campaign.setAgencyId(0L);

            var servicedInfo = servicedInfos.get(i);
            if (servicedInfo.getIsServiced()) {
                campaign.setManagerUid(servicedInfo.getManagerUid());
            } else if (agencyUidAndClientId != null) {
                Long agencyUid = agencyUidAndClientId.getUid();
                Optional<UserPerminfo> limitedPerm = ppcRbac.getUserPerminfo(agencyUidAndClientId.getUid())
                        .filter(perm -> perm.hasRepType(RbacRepType.LIMITED));
                if (limitedPerm.isPresent() && featureService.isEnabledForClientId(
                        agencyUidAndClientId.getClientId(),
                        FeatureName.NEW_LIM_REP_SCHEMA)) {
                    agencyUid = client.getAgencyUserId();
                }
                campaign.setAgencyUid(agencyUid);
                campaign.setAgencyId(agencyUidAndClientId.getClientId().asLong());
            }

            campaign.setIsServiceRequested(false);

            campaign.setCurrency(client.getWorkCurrency());

            setDefaultStatuses(addCampaignParametersContainer, campaign);
            CampaignConverter.setFieldIfNull(campaign, CommonCampaign.ENABLE_SEND_ACCOUNT_NEWS,
                    CampaignConstants.DEFAULT_ENABLE_SEND_ACCOUNT_NEWS);
            CampaignConverter.setFieldIfNull(campaign, CommonCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT,
                    CampaignConstants.DEFAULT_ENABLE_PAUSED_BY_DAY_BUDGET_EVENT);
            CampaignConverter.setFieldIfNull(campaign, CommonCampaign.WARNING_BALANCE,
                    CampaignConstants.DEFAULT_CAMPAIGN_WARNING_BALANCE);

            campaign.setLastChange(NOW_PLACEHOLDER);
        }
    }

    private void setDefaultStatuses(RestrictedCampaignsAddOperationContainer addOperationContainer,
                                    CommonCampaign campaign) {
        campaign.setStatusActive(false);
        campaign.setStatusEmpty(true);
        campaign.setStatusArchived(false);
        campaign.setStatusBsSynced(CampaignStatusBsSynced.NO);
        campaign.setStatusShow(calculateStatusShow(addOperationContainer, campaign));
        campaign.setStatusModerate(calculateStatusModerate(addOperationContainer, campaign));
        campaign.setStatusPostModerate(calculateStatusPostModerate(addOperationContainer, campaign));
    }

    private Boolean calculateStatusShow(
            RestrictedCampaignsAddOperationContainer addOperationContainer, CommonCampaign campaign) {
        // кампании внутренней рекламы по дефолту создаем выключенными
        if (CampaignTypeKinds.INTERNAL.contains(campaign.getType())) {
            return false;
        }

        // используем статус, установленный в операции копирования
        if (addOperationContainer.isCopy()) {
            return campaign.getStatusShow();
        }

        return true;
    }

    private CampaignStatusModerate calculateStatusModerate(
            RestrictedCampaignsAddOperationContainer addOperationContainer, CommonCampaign campaign) {
        if (CampaignTypeKinds.INTERNAL.contains(campaign.getType())) {
            return CampaignStatusModerate.READY;
        }

        // используем статус, установленный в операции копирования
        if (addOperationContainer.isCopy()) {
            return campaign.getStatusModerate();
        }

        if (addOperationContainer.getOptions().getKeepModerationStatuses()
                && campaign.getStatusModerate() != null) {
            return campaign.getStatusModerate();
        }

        if (addOperationContainer.isReadyToModerate()) {
            return CampaignStatusModerate.READY;
        }

        return CampaignStatusModerate.NEW;
    }

    private CampaignStatusPostmoderate calculateStatusPostModerate(
            RestrictedCampaignsAddOperationContainer addOperationContainer, CommonCampaign campaign) {
        if (addOperationContainer.getOptions().getKeepModerationStatuses()
                && campaign.getStatusModerate() != null) {
            return campaign.getStatusPostModerate();
        }

        if (isValidId(campaign.getManagerUid()) || isValidId(campaign.getAgencyUid())) {
            // allow pay without blocking for serviced campaigns
            return CampaignStatusPostmoderate.ACCEPTED;
        }

        return CampaignStatusPostmoderate.NEW;
    }

    private void setProductId(Client client, CommonCampaign campaign) {
        if (campaign != null && campaign.getType() == CampaignType.CPM_PRICE) {
            //для прайсовых кампаний зависит от пакета.
            //Логика реализована в специфичном саппорте CampaignWithPricePackageAddOperationSupport
            //В этом классе не хотим перезатирать
            return;
        }

        campaign.setProductId(productService.calculateProductId(client, campaign));
    }

    private void setZeroSumValues(CommonCampaign campaign) {
        campaign.setSum(BigDecimal.ZERO);
        campaign.setSumToPay(BigDecimal.ZERO);
        campaign.setSumSpent(BigDecimal.ZERO);
        campaign.setSumLast(BigDecimal.ZERO);
        campaign.setOrderId(0L);
    }

    @Override
    public void beforeExecution(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                List<CommonCampaign> campaigns) {
        if (campaigns.stream().map(CommonCampaign::getType).noneMatch(CampaignTypeKinds.UNDER_WALLET::contains)) {
            return;
        }

        Long walletId = walletService.createWalletIfNeeded(addCampaignParametersContainer);
        if (walletId != null) {
            campaigns.forEach(c -> c.setWalletId(walletId));
        }
    }

    @Override
    public void afterExecution(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                               List<CommonCampaign> campaigns) {
        Client client = clientService.getClient(addCampaignParametersContainer.getClientId());

        List<AddServicedCampaignInfo> servicedInfos =
                addServicedCampaignService.getServicedInfoForNewCampaigns(addCampaignParametersContainer, campaigns);

        @Nullable UidAndClientId agencyUidAndClientId = userService.getAgencyUidAndClientId(
                addCampaignParametersContainer.getOperatorUid(), client.getAgencyUserId(), client.getAgencyClientId());
        updateClient(client, agencyUidAndClientId, servicedInfos);

        createBillingAggregatesIfNeeded(addCampaignParametersContainer, campaigns);

        if (addCampaignParametersContainer.getOptions().isSendAutoServicingMailNotification()) {
            sendAutoServicingMailNotificationsIfNeeded(addCampaignParametersContainer, campaigns);
        }
    }

    private void createBillingAggregatesIfNeeded(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                                 List<CommonCampaign> campaigns) {
        Long walletId = StreamEx.of(campaigns)
                .map(CommonCampaign::getWalletId)
                .nonNull()
                .findAny()
                .orElse(0L);
        if (isValidId(walletId)) {
            billingAggregateService.createBillingAggregates(addCampaignParametersContainer.getClientId(),
                    addCampaignParametersContainer.getOperatorUid(),
                    addCampaignParametersContainer.getChiefUid(),
                    campaigns,
                    walletId);
        }
    }

    private void sendAutoServicingMailNotificationsIfNeeded(
            RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
            List<CommonCampaign> campaigns) {

        var campaignsWithManagers = filterList(campaigns, x -> x.getManagerUid() != null);
        if (campaignsWithManagers.size() > 0) {
            var subjectUser = userService.getUser(addCampaignParametersContainer.getClientUid());
            var managerUids = listToSet(campaignsWithManagers, CommonCampaign::getManagerUid);
            checkState(managerUids.size() == 1,
                    "More than one manager in the single request is not supported");
            var managerUid = managerUids.iterator().next();
            var manager = userService.getUser(managerUid);

            campaignsWithManagers.forEach(campaign -> {
                var notification = new AutoServicingMailNotification(
                        campaign.getId(),
                        campaign.getName(),
                        campaign.getType(),
                        subjectUser,
                        manager);
                notificationService.addNotification(notification);
            });
        }

    }

    /**
     * Обновляется клиент и добавляется связь с мэнэджером, если необходимо
     */
    private void updateClient(
            Client client,
            @Nullable UidAndClientId agencyUidAndClientId,
            List<AddServicedCampaignInfo> servicedInfos) {

        AppliedChanges<Client> clientAppliedChanges = getClientAppliedChanges(client, agencyUidAndClientId);

        clientService.update(clientAppliedChanges);

        servicedInfos.stream()
                .filter(AddServicedCampaignInfo::getIsServiced)
                .findAny()
                .ifPresent(servicedInfo -> {
                    //noinspection ConstantConditions managerUid nonNull when isServices == true
                    clientService.bindClientsToManager(servicedInfo.getManagerUid(),
                            List.of(ClientId.fromLong(client.getClientId())));
                });
    }

    private AppliedChanges<Client> getClientAppliedChanges(
            Client client,
            @Nullable UidAndClientId agencyUidAndClientId) {
        ModelChanges<Client> clientModelChanges = new ModelChanges<>(client.getId(), Client.class);
        if (client.getRole().anyOf(RbacRole.EMPTY)) {
            clientModelChanges.process(RbacRole.CLIENT, Client.ROLE);
        }

        if (agencyUidAndClientId != null) {
            clientModelChanges.process(agencyUidAndClientId.getClientId().asLong(), Client.AGENCY_CLIENT_ID);
            UserPerminfo agencyUserPerminfo = ppcRbac.getUserPerminfo(agencyUidAndClientId.getUid())
                    .orElseThrow(() -> new IllegalStateException("agencyUid" + agencyUidAndClientId.getUid() +
                            " has not perm info"));
            if (agencyUserPerminfo.hasRepType(RbacRepType.LIMITED)) {
                // В новой схеме не нужно проставлять в clients.agancy_uid ограниченного представителя
                if (!featureService.isEnabledForClientId(
                        agencyUidAndClientId.getClientId(),
                        FeatureName.NEW_LIM_REP_SCHEMA)) {
                    clientModelChanges.process(agencyUidAndClientId.getUid(), Client.AGENCY_USER_ID);
                }
            } else {
                clientModelChanges.process(agencyUserPerminfo.chiefUid(), Client.AGENCY_USER_ID);
            }
        }

        return clientModelChanges.applyTo(client);
    }

}
