package ru.yandex.direct.core.entity.campaign.service.type.update

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.model.BannerWithButton
import ru.yandex.direct.core.entity.banner.model.BannerWithHref
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerCampaignIdFilter
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory
import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBannerHrefParams
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.result.MassResult
import java.util.function.BiFunction

/**
 * Саппорт для обновления utm-шаблона на кампании.
 * При изменении шаблона выполняет виртуальные обновления связанных сущностей (банер, кнопка, сайтлинки),
 * чем триггерит пересчет статусов модерации и синхронизации с БК.
 */
@Component
class CampaignWithBannerHrefParamsUpdateOperationSupport(
    private val bannerTypedRepository: BannerTypedRepository,
    private val bannersUpdateOperationFactory: BannersUpdateOperationFactory
) : AbstractCampaignUpdateOperationSupport<CampaignWithBannerHrefParams>() {

    companion object {
        private val logger = LoggerFactory.getLogger(CampaignWithBannerHrefParamsUpdateOperationSupport::class.java)
        // кастомная функция сравнения, которая считает лючые значения разными
        // нужна для того, чтобы UpdateOperation считал изменение значения на такое же реальным обновлением
        private val CUSTOM_ANY_NOT_EQUALS = BiFunction { _: Any, _: Any -> false }
    }

    override fun getTypeClass(): Class<CampaignWithBannerHrefParams> {
        return CampaignWithBannerHrefParams::class.java
    }

    override fun addToAdditionalActionsContainer(
        additionalActionsContainer: CampaignAdditionalActionsContainer,
        updateContainer: RestrictedCampaignsUpdateOperationContainer,
        appliedChanges: MutableList<AppliedChanges<CampaignWithBannerHrefParams>>
    ) {
        val campaignIdsWithChangedHrefParams = getCampaignIdsWithChangedHrefParams(appliedChanges)
        additionalActionsContainer.resetCampaignBannersStatusBsSynced(campaignIdsWithChangedHrefParams)
        additionalActionsContainer.resetCampaignAdGroupsStatusBsSynced(campaignIdsWithChangedHrefParams)
    }

    override fun updateRelatedEntitiesOutOfTransaction(
        updateContainer: RestrictedCampaignsUpdateOperationContainer,
        appliedChanges: MutableList<AppliedChanges<CampaignWithBannerHrefParams>>
    ) {
        val campaignIdsWithChangedHrefParams = appliedChanges.asSequence()
            .filter { it.changed(CampaignWithBannerHrefParams.BANNER_HREF_PARAMS) }
            .map { it.model.id }
            .toSet()

        val banners = bannerTypedRepository.getSafely(
            updateContainer.shard, bannerCampaignIdFilter(campaignIdsWithChangedHrefParams),
            listOf(BannerWithHref::class.java, BannerWithButton::class.java, BannerWithSitelinks::class.java)
        )

        resetBannerHrefs(updateContainer, banners.filterIsInstance<BannerWithHref>())
        resetButtonHrefs(updateContainer, banners.filterIsInstance<BannerWithButton>())
        resetSitelinks(updateContainer, banners.filterIsInstance<BannerWithSitelinks>())
    }

    private fun resetSitelinks(
        container: RestrictedCampaignsUpdateOperationContainer,
        banners: List<BannerWithSitelinks>
    ) {
        val modelChanges = banners
            .filter { it.sitelinksSetId != null }
            .map {
                ModelChanges(it.id, BannerWithSitelinks::class.java)
                    .process(it.sitelinksSetId, BannerWithSitelinks.SITELINKS_SET_ID)
            }
        val updateResult = bannersUpdateOperationFactory.createPartialUpdateOperationWithCustomModelEquals(
            modelChanges, container.operatorUid, container.clientId, container.shard, BannerWithSitelinks::class.java,
            mapOf(BannerWithSitelinks.SITELINKS_SET_ID to CUSTOM_ANY_NOT_EQUALS)
        ).prepareAndApply()
        logUpdateResult(updateResult, "reset banner sitelink sets")
    }

    private fun resetBannerHrefs(
        container: RestrictedCampaignsUpdateOperationContainer,
        banners: List<BannerWithHref>
    ) {
        val modelChanges = banners
            .filter { it.href != null }
            .map {
                ModelChanges(it.id, BannerWithHref::class.java)
                    .process(it.href, BannerWithHref.HREF)
            }
        val updateResult = bannersUpdateOperationFactory.createPartialUpdateOperationWithCustomModelEquals(
            modelChanges, container.operatorUid, container.clientId, container.shard, BannerWithHref::class.java,
            mapOf(BannerWithHref.HREF to CUSTOM_ANY_NOT_EQUALS)
        ).prepareAndApply()
        logUpdateResult(updateResult, "reset banner hrefs")
    }

    private fun resetButtonHrefs(
        container: RestrictedCampaignsUpdateOperationContainer,
        banners: List<BannerWithButton>
    ) {
        val modelChanges = banners
            .filter { it.buttonHref != null }
            .map {
                ModelChanges(it.id, BannerWithButton::class.java)
                    .process(it.buttonHref, BannerWithButton.BUTTON_HREF)
            }
        val updateResult = bannersUpdateOperationFactory.createPartialUpdateOperationWithCustomModelEquals(
            modelChanges, container.operatorUid, container.clientId, container.shard, BannerWithButton::class.java,
            mapOf(BannerWithButton.BUTTON_HREF to CUSTOM_ANY_NOT_EQUALS)
        ).prepareAndApply()
        logUpdateResult(updateResult, "reset button hrefs")
    }

    private fun logUpdateResult(result: MassResult<Long>, update: String) {
        val successfulCount = result.successfulCount
        val errorCount = result.errorCount
        logger.info(
            "Result of $update for ${successfulCount + errorCount} banners: " +
                "$successfulCount OK, " +
                "$errorCount ERROR due to defects ${result.errors.map { it.defect.defectId() }}"
        )
    }

    private fun getCampaignIdsWithChangedHrefParams(
        appliedChanges: List<AppliedChanges<CampaignWithBannerHrefParams>>
    ): Set<Long> {
        return appliedChanges.asSequence()
            .filter { it.changed(CampaignWithBannerHrefParams.BANNER_HREF_PARAMS) }
            .map { it.model.id }
            .toSet()
    }
}
