package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBidModifiers;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.type.CampaignWithBidModifiersUtils;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBidModifiersUpdateOperationSupport
        extends AbstractCampaignUpdateOperationSupport<CampaignWithBidModifiers> {

    private final BidModifierService bidModifierService;
    private final FeatureService featureService;

    @Autowired
    public CampaignWithBidModifiersUpdateOperationSupport(BidModifierService bidModifierService,
                                                          FeatureService featureService) {
        this.bidModifierService = bidModifierService;
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithBidModifiers> getTypeClass() {
        return CampaignWithBidModifiers.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer container,
                                 List<AppliedChanges<CampaignWithBidModifiers>> appliedChanges) {
        List<AppliedChanges<CampaignWithBidModifiers>> appliedChangesWithChangedBidModifiers =
                filterList(appliedChanges, ac -> ac.changed(CampaignWithBidModifiers.BID_MODIFIERS));

        appliedChangesWithChangedBidModifiers
                .forEach(CampaignWithBidModifiersUpdateOperationSupport::fillBidModifier);

        if (!featureService.isEnabledForClientId(container.getClientId(),
                FeatureName.GEO_RATE_CORRECTIONS_ALLOWED_FOR_DNA)) {
            appliedChangesWithChangedBidModifiers
                    .forEach(CampaignWithBidModifiersUpdateOperationSupport::addOldGeoBidModifier);
        }
    }

    /**
     * Из нового интерфейса нельзя менять геотаргетинг на кампанию.
     * Поэтому если у нас уже есть геокорректировка добавляем её в новые значения.
     */
    private static void addOldGeoBidModifier(AppliedChanges<CampaignWithBidModifiers> campaignChanges) {
        List<BidModifier> oldBidModifiers = campaignChanges.getOldValue(CampaignWithBidModifiers.BID_MODIFIERS);

        if (oldBidModifiers == null) {
            return;
        }

        Optional<BidModifier> oldGeoBidModifier = oldBidModifiers.stream()
                .filter(BidModifierGeo.class::isInstance)
                .findAny();

        if (oldGeoBidModifier.isPresent()) {
            List<BidModifier> newBidModifiers = campaignChanges.getNewValue(CampaignWithBidModifiers.BID_MODIFIERS);

            List<BidModifier> newBidModifiersWithOldGeo = isEmpty(newBidModifiers)
                    ? List.of(oldGeoBidModifier.get())
                    : ListUtils.union(List.of(oldGeoBidModifier.get()), newBidModifiers);

            campaignChanges.modify(CampaignWithBidModifiers.BID_MODIFIERS, newBidModifiersWithOldGeo);
        }
    }

    private static void fillBidModifier(AppliedChanges<CampaignWithBidModifiers> campaignChanges) {
        if (campaignChanges.getModel().getBidModifiers() == null) {
            return;
        }
        campaignChanges.getModel().getBidModifiers()
                .forEach(bm -> bm.setCampaignId(campaignChanges.getModel().getId()));

        enrichBidModifiersIfNeeded(campaignChanges);
    }

    private static void enrichBidModifiersIfNeeded(AppliedChanges<CampaignWithBidModifiers> campaignChanges) {

        // Добавляются корректировки только для Медийных кампаний
        if (campaignChanges.getModel().getType() != CampaignType.CPM_BANNER) {
            return;
        }
        CampaignWithBidModifiersUtils.addInBannerToInAppBidModifiers(campaignChanges.getModel().getBidModifiers());
    }

    @Override
    public void updateRelatedEntitiesInTransaction(DSLContext dslContext,
                                                   RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                   List<AppliedChanges<CampaignWithBidModifiers>> appliedChanges) {
        List<CampaignWithBidModifiers> campaignWithBidModifiersList = filterAndMapList(
                appliedChanges,
                ac -> ac.changed(CampaignWithBidModifiers.BID_MODIFIERS),
                AppliedChanges::getModel);

        List<BidModifier> bidModifiers = StreamEx.of(campaignWithBidModifiersList)
                .flatCollection(CampaignWithBidModifiers::getBidModifiers)
                .nonNull()
                .toList();

        bidModifierService.replaceCampaignModifiers(dslContext, updateParameters.getClientId(),
                updateParameters.getOperatorUid(), bidModifiers,
                listToSet(campaignWithBidModifiersList, CampaignWithBidModifiers::getId)
        );
    }

}
