package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.brandSurvey.BrandSurvey;
import ru.yandex.direct.core.entity.campaign.model.CampaignExperiment;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandLiftExperimentsService;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.ExperimentRetargetingConditions;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.CollectionUtils;

import static ru.yandex.direct.feature.FeatureName.BRAND_LIFT_HIDDEN;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBrandLiftUpdateOperationSupport extends AbstractCampaignUpdateOperationSupport<CampaignWithBrandLift> {

    private final RetargetingConditionService retargetingConditionService;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;
    private final CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService;

    @Autowired
    public CampaignWithBrandLiftUpdateOperationSupport(RetargetingConditionService retargetingConditionService,
                                                       ShardHelper shardHelper,
                                                       FeatureService featureService,
                                                       CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService) {
        this.retargetingConditionService = retargetingConditionService;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.campaignWithBrandLiftExperimentsService = campaignWithBrandLiftExperimentsService;
    }

    @Override
    public Class<CampaignWithBrandLift> getTypeClass() {
        return CampaignWithBrandLift.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                 List<AppliedChanges<CampaignWithBrandLift>> appliedChanges) {
        boolean brandLiftHiddenEnabledForOperator =
                featureService.isEnabled(updateContainer.getOperatorUid(), BRAND_LIFT_HIDDEN);

        appliedChanges.stream()
                .filter(change -> change.changed(CampaignWithBrandLift.BRAND_SURVEY_ID) ||
                        change.changed(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS) ||
                        change.changed(CampaignWithBrandLift.SECTION_IDS))
                .forEach(change -> {
                    boolean isBrandLiftHidden =
                            nvl(change.getOldValue(CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN), false) &&
                                    change.getOldValue(CampaignWithBrandLift.BRAND_SURVEY_ID) != null;
                    boolean isBrandLiftChanged = change.changed(CampaignWithBrandLift.BRAND_SURVEY_ID);
                    boolean isBrandLiftDeleted = change.deleted(CampaignWithBrandLift.BRAND_SURVEY_ID);
                    boolean isExperimentChanged = change.changed(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS) ||
                            change.changed(CampaignWithBrandLift.SECTION_IDS);
                    boolean isExperimentDeleted =
                            change.deleted(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS) &&
                                    (change.deleted(CampaignWithBrandLift.SECTION_IDS) ||
                                            nvl(change.getNewValue(CampaignWithBrandLift.SECTION_IDS),
                                                    Collections.emptyList()).isEmpty());

                    // не даем удалить скрытый Brand Lift при редактировании кампании клиентом, если он не заводит
                    // собственный Brand Lift или собственные эксперименты
                    if (isBrandLiftHidden &&
                            (!isBrandLiftChanged || isBrandLiftDeleted) &&
                            (!isExperimentChanged || isExperimentDeleted)) {
                        if (!brandLiftHiddenEnabledForOperator) {
                            change.modify(CampaignWithBrandLift.BRAND_SURVEY_ID,
                                    change.getOldValue(CampaignWithBrandLift.BRAND_SURVEY_ID));
                        }
                        if (!brandLiftHiddenEnabledForOperator || !isBrandLiftDeleted) {
                            change.modify(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS,
                                    change.getOldValue(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS));
                            change.modify(CampaignWithBrandLift.SECTION_IDS,
                                    change.getOldValue(CampaignWithBrandLift.SECTION_IDS));
                        } else {
                            change.modify(CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN, false);
                        }
                    } else {
                        change.modify(CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN,
                                brandLiftHiddenEnabledForOperator && isBrandLiftChanged && !isBrandLiftDeleted);
                    }
                });
    }

    @Override
    public void beforeExecution(RestrictedCampaignsUpdateOperationContainer updateParameters,
                                List<AppliedChanges<CampaignWithBrandLift>> appliedChanges) {

        Map<Long, List<Long>> preparedExperiments = null;
        var brandLifts =
                campaignWithBrandLiftExperimentsService.clientBrandLifts(updateParameters.getClientId()).stream()
                        .collect(Collectors.toMap(BrandSurvey::getBrandSurveyId, Function.identity()));
        String clientLogin = shardHelper.getLoginByUid(updateParameters.getChiefUid());

        var modelsWithExistedBrandLifts = new ArrayList<AppliedChanges<CampaignWithBrandLift>>();
        List<AppliedChanges<CampaignWithBrandLift>> modelsWithoutExistedBrandLifts = new ArrayList<>();

        for (var campaign : appliedChanges) {
            if (brandLifts.containsKey(campaign.getModel().getBrandSurveyId())) {
                modelsWithExistedBrandLifts.add(campaign);
            } else {
                modelsWithoutExistedBrandLifts.add(campaign);
                brandLifts.put(campaign.getModel().getBrandSurveyId(), null);
            }
        }

        // Если был добавлен BL или SL, создаем новый эксперимент
        var filteredChanges = filterList(
                modelsWithoutExistedBrandLifts, ac ->
                        ac.changedAndNotDeleted(CampaignWithBrandLift.BRAND_SURVEY_ID) ||
                                ac.changedAndNotDeleted(CampaignWithBrandLift.IS_SEARCH_LIFT_ENABLED)
        );

        if (!CollectionUtils.isEmpty(filteredChanges)) {
            preparedExperiments = StreamEx.of(filteredChanges)
                    .flatMapToEntry(ac ->
                                    campaignWithBrandLiftExperimentsService.prepareExperiment(clientLogin, ac.getModel()))
                    .grouping();
        }

        // для кампаний со скрытым Brand-Lift не пересоздаем условия ретаргетинга на аб-эксперименты,
        // так как аб-эксперимент зафиксирован и сегменты добавлять нельзя
        modelsWithoutExistedBrandLifts = modelsWithoutExistedBrandLifts.stream()
                .filter(change -> !nvl(change.getNewValue(CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN), false) ||
                        change.changedAndNotDeleted(CampaignWithBrandLift.BRAND_SURVEY_ID))
                .collect(Collectors.toList());

        var experimentRetargetingConditions = getRetargetingConditions(modelsWithoutExistedBrandLifts,
                updateParameters, preparedExperiments);

        Map<Long, List<Long>> finalPreparedExperiments = preparedExperiments;
        EntryStream.of(modelsWithoutExistedBrandLifts)
                .forKeyValue((index, ac) -> {
                    var retargetingConditions = experimentRetargetingConditions.get(index);
                    updateCampaignWithExperimentRetargetingConditions(
                            ac,
                            retargetingConditions
                    );
                    if (ac.changedAndNotDeleted(CampaignWithBrandLift.BRAND_SURVEY_NAME) ||
                            ac.changedAndNotDeleted(CampaignWithBrandLift.BRAND_SURVEY_ID)) {
                        String brandSurveyName = nvl(ac.getModel().getBrandSurveyName(), "").isBlank() ?
                                "Brand-lift " + ac.getModel().getId() :
                                ac.getModel().getBrandSurveyName();
                        if (ac.changedAndNotDeleted(CampaignWithBrandLift.BRAND_SURVEY_ID)) {

                            var brandSurvey = new BrandSurvey()
                                    .withClientId(updateParameters.getClientId().asLong())
                                    .withBrandSurveyId(ac.getModel().getBrandSurveyId())
                                    .withName(brandSurveyName)
                                    .withRetargetingConditionId(retargetingConditions
                                            .getRetargetingConditionId()
                                    )
                                    .withIsBrandLiftHidden(ac.getModel().getIsBrandLiftHidden())
                                    .withExperimentId(ac.getModel().getExperimentId())
                                    .withSegmentId(Optional.ofNullable(finalPreparedExperiments
                                                    .get(ac.getModel().getExperimentId()))
                                            .map(list -> list.get(0))
                                            .orElse(null));

                            campaignWithBrandLiftExperimentsService.addBrandSurvey(clientLogin, brandSurvey);
                            brandLifts.put(brandSurvey.getBrandSurveyId(), brandSurvey);
                        } else {
                            campaignWithBrandLiftExperimentsService.renameBrandSurvey(clientLogin,
                                    ac.getOldValue(CampaignWithBrandLift.BRAND_SURVEY_ID), brandSurveyName);
                        }
                    }
                });

        Map<Long, List<Long>> preparedExperimentsWithExistedBl = StreamEx.of(modelsWithExistedBrandLifts)
                .filter(ac -> ac.changedAndNotDeleted(CampaignWithBrandLift.BRAND_SURVEY_ID))
                .flatMapToEntry(ac -> {
                    var bl = brandLifts.get(ac.getModel().getBrandSurveyId());
                    ac.getModel().setSectionIds(nvl(ac.getModel().getSectionIds(), new ArrayList<>()));
                    ac.getModel().setAbSegmentGoalIds(nvl(ac.getModel().getAbSegmentGoalIds(), new ArrayList<>()));
                    ac.getModel().getSectionIds().add(nvl(bl.getExperimentId(), 0L));
                    ac.getModel().getAbSegmentGoalIds().add(nvl(bl.getSegmentId(), 0L));
                    return Map.of(nvl(bl.getExperimentId(), 0L), nvl(bl.getSegmentId(), 0L));
                }).distinct().grouping();

        var experimentRetargetingConditionsWithExistedBl = getRetargetingConditions(modelsWithExistedBrandLifts,
                updateParameters, preparedExperimentsWithExistedBl);

        EntryStream.of(modelsWithExistedBrandLifts)
                .forKeyValue((index, ac) -> {
                    var brandSurvey = brandLifts.get(ac.getModel().getBrandSurveyId());
                    updateCampaignWithExperimentRetargetingConditions(ac,
                            experimentRetargetingConditionsWithExistedBl.get(index));

                    if (ac.getModel().getBrandSurveyName() != null &&
                            !brandSurvey.getName().equals(ac.getModel().getBrandSurveyName())) {
                        campaignWithBrandLiftExperimentsService.renameBrandSurvey(
                                clientLogin,
                                brandSurvey.getBrandSurveyId(),
                                ac.getModel().getBrandSurveyName()
                        );
                    }
                });
    }

    private void updateCampaignWithExperimentRetargetingConditions(
            AppliedChanges<CampaignWithBrandLift> campaignAppliedChanges,
            ExperimentRetargetingConditions experimentRetargetingConditions) {
        campaignAppliedChanges.modify(CampaignWithBrandLift.AB_SEGMENT_RETARGETING_CONDITION_ID,
                experimentRetargetingConditions.getRetargetingConditionId());

        campaignAppliedChanges.modify(CampaignWithBrandLift.AB_SEGMENT_STATISTIC_RETARGETING_CONDITION_ID,
                experimentRetargetingConditions.getStatisticRetargetingConditionId());
    }

    private void updateCampaignWithBrandLiftRetargetingConditions(
            AppliedChanges<CampaignWithBrandLift> campaignAppliedChanges, long retargetingConditionId
    ) {
        campaignAppliedChanges.modify(CampaignWithBrandLift.AB_SEGMENT_RETARGETING_CONDITION_ID,
                retargetingConditionId);

        campaignAppliedChanges.modify(CampaignWithBrandLift.AB_SEGMENT_STATISTIC_RETARGETING_CONDITION_ID,
                retargetingConditionId);
    }

    private List<ExperimentRetargetingConditions> getRetargetingConditions(List<AppliedChanges<CampaignWithBrandLift>> campaigns,
                                                        RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                        @Nullable  Map<Long, List<Long>> preparedExperiments) {
        if (CollectionUtils.isEmpty(campaigns)) {
            return null;
        }
        List<CampaignExperiment> campaignExperimentsWithExistedBl = mapList(
                campaigns,
                item -> new CampaignExperiment()
                        .withAbSegmentGoalIds(item.getModel().getAbSegmentGoalIds())
                        .withSectionIds(item.getModel().getSectionIds()));

        return retargetingConditionService.findOrCreateExperimentsRetargetingConditions(
                        updateParameters.getClientId(),
                        campaignExperimentsWithExistedBl,
                        updateParameters.getMetrikaClient().getAbSegmentGoals(),
                        preparedExperiments);
    }
}
