package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBroadMatch;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.BroadMatchUncheckNotification;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBroadMatchUpdateOperationSupport extends AbstractCampaignUpdateOperationSupport<CampaignWithBroadMatch> {

    private final NotificationService notificationService;
    private final UserRepository userRepository;
    private final RbacService rbacService;
    private final AutobudgetAlertService autobudgetAlertService;

    @Autowired
    public CampaignWithBroadMatchUpdateOperationSupport(NotificationService notificationService,
                                                        UserRepository userRepository, RbacService rbacService,
                                                        AutobudgetAlertService autobudgetAlertService) {
        this.notificationService = notificationService;
        this.userRepository = userRepository;
        this.rbacService = rbacService;
        this.autobudgetAlertService = autobudgetAlertService;
    }

    @Override
    public Class<CampaignWithBroadMatch> getTypeClass() {
        return CampaignWithBroadMatch.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer container,
                                 List<AppliedChanges<CampaignWithBroadMatch>> appliedChanges) {
        appliedChanges.forEach(changes -> {
            CampaignWithBroadMatch campaign = changes.getModel();

            DbStrategy strategy = campaign.getStrategy();
            BroadMatch broadMatch = campaign.getBroadMatch();

            // если стратегия "только на сети" выключаем ДРФ: DIRECT-68661
            if (strategy.isSearchStop() && broadMatch.getBroadMatchFlag()) {
                changes.modify(CampaignWithBroadMatch.BROAD_MATCH, new BroadMatch()
                        .withBroadMatchFlag(false)
                        .withBroadMatchLimit(broadMatch.getBroadMatchLimit())
                        .withBroadMatchGoalId(broadMatch.getBroadMatchGoalId()));
            }
        });
    }

    @Override
    public void addToAdditionalActionsContainer(CampaignAdditionalActionsContainer additionalActionsContainer,
                                                RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                List<AppliedChanges<CampaignWithBroadMatch>> appliedChanges) {
        additionalActionsContainer.addCampaignsToFreezeAlerts(
                autobudgetAlertService.getCampaignsToFreezeAlertsOnBroadMatchChange(
                        updateParameters.getClientId(),
                        appliedChanges)
        );
    }

    @Override
    public void updateRelatedEntitiesOutOfTransaction(RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                      List<AppliedChanges<CampaignWithBroadMatch>> appliedChanges) {
        broadMatchUncheckNotification(updateParameters, appliedChanges);
    }

    private void broadMatchUncheckNotification(RestrictedCampaignsUpdateOperationContainer updateParameters,
                                               List<AppliedChanges<CampaignWithBroadMatch>> appliedChanges) {

        RbacRole operatorRole = rbacService.getUidRole(updateParameters.getOperatorUid());
        int shard = updateParameters.getShard();

        List<CampaignWithBroadMatch> campaignsForNotify = filterAndMapList(appliedChanges, ac -> {

            CampaignWithBroadMatch campaign = ac.getModel();
            Long managerUid = campaign.getManagerUid();
            DbStrategy strategy = campaign.getStrategy();

            BroadMatch oldBroadMatch = ac.getOldValue(CampaignWithBroadMatch.BROAD_MATCH);
            BroadMatch newBroadMatch = ac.getNewValue(CampaignWithBroadMatch.BROAD_MATCH);

            // Если есть менеджер у кампании и действия производит обычный клиент
            if (managerUid != null && operatorRole == RbacRole.CLIENT) {
                // и стратегия включена на поиске
                if (!strategy.isSearchStop() || !strategy.isDifferentPlaces()) {
                    // и broad_match_flag был раньше взведен, а теперь нет
                    return oldBroadMatch.getBroadMatchFlag() && !newBroadMatch.getBroadMatchFlag();
                }
            }

            return false;

        }, AppliedChanges::getModel);

        if (campaignsForNotify.isEmpty()) {
            return;
        }

        List<User> users = userRepository.fetchByUids(shard, List.of(updateParameters.getChiefUid()));
        checkArgument(!users.isEmpty(), "User not found, uid=" + updateParameters.getChiefUid() + "," +
                " shard=" + shard);
        String clientLogin = users.get(0).getLogin();

        campaignsForNotify.forEach(campaign -> {
            BroadMatchUncheckNotification notification = new BroadMatchUncheckNotification()
                    .withCampaignId(campaign.getId())
                    .withManagerUid(campaign.getManagerUid())
                    .withClientId(updateParameters.getClientId())
                    .withClientLogin(clientLogin);

            notificationService.addNotification(notification);
        });
    }
}
