package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.banner.type.vcard.BannerWithVcardUtils;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithContactInfo;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.core.entity.vcard.repository.internal.OrgDetailsRepository;
import ru.yandex.direct.core.entity.vcard.service.VcardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.AddedModelId;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.Collectors.nullFriendlyMapCollector;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithContactInfoUpdateOperationSupport extends AbstractCampaignUpdateOperationSupport<CampaignWithContactInfo> {
    private final VcardHelper vcardHelper;
    private final CampaignModifyRepository campaignModifyRepository;
    private final BannerService newBannerService;
    private final VcardRepository vcardRepository;
    private final BannerTypedRepository bannerRepository;
    private final OrgDetailsRepository orgDetailsRepository;

    @Autowired
    public CampaignWithContactInfoUpdateOperationSupport(VcardHelper vcardHelper,
                                                         CampaignModifyRepository campaignModifyRepository,
                                                         BannerService newBannerService,
                                                         VcardRepository vcardRepository,
                                                         BannerTypedRepository bannerRepository,
                                                         OrgDetailsRepository orgDetailsRepository) {
        this.vcardHelper = vcardHelper;
        this.campaignModifyRepository = campaignModifyRepository;
        this.newBannerService = newBannerService;
        this.vcardRepository = vcardRepository;
        this.bannerRepository = bannerRepository;
        this.orgDetailsRepository = orgDetailsRepository;
    }

    @Override
    public Class<CampaignWithContactInfo> getTypeClass() {
        return CampaignWithContactInfo.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer container,
                                 List<AppliedChanges<CampaignWithContactInfo>> appliedChanges) {
        var changesWithVcards = StreamEx.of(appliedChanges)
                .filter(changes -> changes.changed(CampaignWithContactInfo.CONTACT_INFO))
                .toList();

        Map<Long, Vcard> vcardByCampaignId = StreamEx.of(changesWithVcards)
                .map(AppliedChanges::getModel)
                .mapToEntry(CampaignWithContactInfo::getId, CampaignWithContactInfo::getContactInfo)
                .nonNullValues()
                .toMap();

        Collection<Vcard> vcards = vcardByCampaignId.values();

        if (!vcards.isEmpty()) {
            vcardHelper.fillVcardsWithGeocoderData(vcards);
            vcardHelper.fillVcardsWithRegionIds(vcards);
        }

        changesWithVcards.forEach(c -> c.modify(CampaignWithContactInfo.CONTACT_INFO,
                vcardByCampaignId.get(c.getModel().getId())));
    }

    @Override
    public void updateRelatedEntitiesOutOfTransactionWithModelChanges(
            RestrictedCampaignsUpdateOperationContainer updateParameters,
            List<ModelChanges<CampaignWithContactInfo>> modelChanges,
            List<AppliedChanges<CampaignWithContactInfo>> appliedChanges) {
        if (modelChanges == null) {
            return;
        }

        var idsWithVcardChangeRequested = filterAndMapToSet(modelChanges,
                c -> c.isPropChanged(CampaignWithContactInfo.CONTACT_INFO), ModelChanges::getId);

        var changesWithVcards = filterList(appliedChanges,
                c -> idsWithVcardChangeRequested.contains(c.getModel().getId()));

        Set<Long> campaignIds = listToSet(changesWithVcards, c -> c.getModel().getId());

        List<BannerWithSystemFields> campaignsBanners =
                bannerRepository.getBannersByCampaignIdsAndClass(updateParameters.getShard(), campaignIds,
                        BannerWithSystemFields.class);
        List<BannerWithSystemFields> notFilteredCampaignsBanners = filterList(campaignsBanners,
                banner -> !banner.getStatusArchived());

        deleteVcardsIdsFromBanners(updateParameters, changesWithVcards, notFilteredCampaignsBanners);
        createVcards(updateParameters, changesWithVcards, notFilteredCampaignsBanners);

        campaignModifyRepository.cleanOrgDetails(updateParameters.getShard(), updateParameters.getChiefUid());
    }

    private void deleteVcardsIdsFromBanners(RestrictedCampaignsUpdateOperationContainer updateParameters,
                                            Collection<AppliedChanges<CampaignWithContactInfo>> appliedChanges,
                                            List<BannerWithSystemFields> campaignsBanners) {
        Set<Long> campaignsIdsToDeleteContactInfo = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .filter(campaign -> campaign.getContactInfo() == null)
                .map(CampaignWithContactInfo::getId)
                .toSet();

        List<BannerWithVcard> bannersToDeleteVcard = StreamEx.of(campaignsBanners)
                .select(BannerWithVcard.class)
                .mapToEntry(BannerWithVcard::getCampaignId)
                .filterValues(campaignsIdsToDeleteContactInfo::contains)
                .keys()
                .toList();

        if (!bannersToDeleteVcard.isEmpty()) {
            deleteVcardsIdsFromBanners(updateParameters, campaignsIdsToDeleteContactInfo,
                    bannersToDeleteVcard);
        }
    }

    private void deleteVcardsIdsFromBanners(RestrictedCampaignsUpdateOperationContainer updateParameters,
                                            Set<Long> campaignsIdsToDeleteContactInfo,
                                            List<BannerWithVcard> bannersToDeleteVcard) {

        Map<Long, Long> vcardIdByBannerId = StreamEx.of(bannersToDeleteVcard)
                .mapToEntry(BannerWithVcard::getCampaignId, Function.identity())
                .collapseKeys()
                .filterKeys(campaignsIdsToDeleteContactInfo::contains)
                .filterValues(BannerWithVcardUtils::hasCommonVcard)
                .values()
                .flatMap(Collection::stream)
                .mapToEntry(Banner::getId, banner -> (Long) null)
                .collect(nullFriendlyMapCollector());

        if (!vcardIdByBannerId.isEmpty()) {
            newBannerService.updateBannerVcardIds(updateParameters.getShard(),
                    updateParameters.getOperatorUid(),
                    updateParameters.getClientId(),
                    List.copyOf(vcardIdByBannerId.keySet()),
                    vcardIdByBannerId);
        }
    }

    private void createVcards(RestrictedCampaignsUpdateOperationContainer updateParameters,
                              Collection<AppliedChanges<CampaignWithContactInfo>> appliedChanges,
                              List<BannerWithSystemFields> campaignsBanners) {
        List<CampaignWithContactInfo> campaignWithContactInfoList = filterAndMapList(
                appliedChanges,
                ac -> ac.changed(CampaignWithContactInfo.CONTACT_INFO),
                AppliedChanges::getModel);

        Set<Long> campaignIdsWithChangedContactInfo = listToSet(campaignWithContactInfoList,
                CampaignWithContactInfo::getId);

        List<Vcard> vcards = StreamEx.of(campaignWithContactInfoList)
                .map(CampaignWithContactInfo::getContactInfo)
                .nonNull()
                .toList();

        orgDetailsRepository.getOrCreateOrgDetails(updateParameters.getShard(), updateParameters.getClientUid(),
                updateParameters.getClientId(), vcards);

        List<BannerWithVcard> bannersToModifyVcard = StreamEx.of(campaignsBanners)
                .select(BannerWithVcard.class)
                .mapToEntry(BannerWithVcard::getCampaignId)
                .filterValues(campaignIdsWithChangedContactInfo::contains)
                .keys()
                .toList();

        if (!bannersToModifyVcard.isEmpty()) {
            createVcards(updateParameters, campaignWithContactInfoList, bannersToModifyVcard);
        }
    }

    private void createVcards(RestrictedCampaignsUpdateOperationContainer updateParameters,
                              List<CampaignWithContactInfo> campaignWithContactInfoList,
                              List<BannerWithVcard> bannersToAddVcard) {

        List<Vcard> vcardsToAdd = StreamEx.of(campaignWithContactInfoList)
                .mapToEntry(CampaignWithContactInfo::getId, CampaignWithContactInfo::getContactInfo)
                .nonNullValues()
                .mapKeyValue((campaignId, vcard) -> vcard.withCampaignId(campaignId))
                .map(vcard -> vcard.withUid(updateParameters.getChiefUid()))
                .toList();
        VcardHelper.fillSystemDateTimeFields(vcardsToAdd);

        //todo DIRECT-103999: переиспользовать dslContext
        List<AddedModelId> vcardIds = vcardRepository.addVcards(updateParameters.getShard(),
                updateParameters.getChiefUid(),
                updateParameters.getClientId(),
                vcardsToAdd);

        checkState(Objects.equals(vcardIds.size(), vcardsToAdd.size()), "vcardIds should have same size as " +
                "vcardsToAdd");

        Map<Long, Long> vcardIdByCampaignId = EntryStream.of(vcardIds)
                .mapKeys(vcardsToAdd::get)
                .mapKeys(Vcard::getCampaignId)
                .mapValues(AddedModelId::getId)
                .toMap();

        Map<Long, Long> vcardIdByBannerId = StreamEx.of(bannersToAddVcard)
                .mapToEntry(BannerWithVcard::getId, BannerWithVcard::getCampaignId)
                .mapValues(vcardIdByCampaignId::get)
                .nonNullValues()
                .toMap();

        if (!vcardIdByBannerId.isEmpty()) {
            newBannerService.updateBannerVcardIds(updateParameters.getShard(),
                    updateParameters.getOperatorUid(),
                    updateParameters.getClientId(),
                    mapList(bannersToAddVcard, Banner::getId),
                    vcardIdByBannerId);
        }
    }

}
