package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.StatusAutobudgetShow;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.campaign.model.CampaignDayBudgetNotificationStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomDayBudget;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.mailnotification.model.CampaignEvent;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.NumberUtils;

import static ru.yandex.direct.common.util.RepositoryUtils.NOW_PLACEHOLDER;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.NumberUtils.greaterThanZero;
import static ru.yandex.direct.utils.NumberUtils.isZero;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithCustomDayBudgetUpdateOperationSupport extends AbstractCampaignUpdateOperationSupport<CampaignWithCustomDayBudget> {

    private final AdGroupRepository adGroupRepository;

    @Autowired
    public CampaignWithCustomDayBudgetUpdateOperationSupport(AdGroupRepository adGroupRepository) {
        this.adGroupRepository = adGroupRepository;
    }

    @Override
    public Class<CampaignWithCustomDayBudget> getTypeClass() {
        return CampaignWithCustomDayBudget.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer container,
                                 List<AppliedChanges<CampaignWithCustomDayBudget>> appliedChanges) {
        //Если поменялось значение дневного бюджета, и при этом стало не 0 — увеличиваем счетчик кол-ва изменений ДБ
        //за день. Если ли же ДБ выключают, то проставляем режим показа в default
        appliedChanges.forEach(changes -> {
            if (changes.changed(CampaignWithCustomDayBudget.DAY_BUDGET)) {
                if (NumberUtils.greaterThanZero(changes.getModel().getDayBudget())) {
                    changes.modify(CampaignWithCustomDayBudget.DAY_BUDGET_DAILY_CHANGE_COUNT,
                            changes.getModel().getDayBudgetDailyChangeCount() + 1);
                } else {
                    changes.modify(CampaignWithCustomDayBudget.DAY_BUDGET_SHOW_MODE, DayBudgetShowMode.DEFAULT_);
                }

                changes.modify(CampaignWithCustomDayBudget.DAY_BUDGET_LAST_CHANGE, NOW_PLACEHOLDER);
            }

            //NB: для wallet'а немного иначе
            if (shouldResetDayBudgetNotificationStatus(changes)) {
                changes.modify(CampaignWithCustomDayBudget.DAY_BUDGET_STOP_TIME, null);
                changes.modify(CampaignWithCustomDayBudget.DAY_BUDGET_NOTIFICATION_STATUS,
                        CampaignDayBudgetNotificationStatus.READY);
            }
        });
    }


    static boolean shouldResetDayBudgetNotificationStatus(AppliedChanges<CampaignWithCustomDayBudget> ac) {
        if (ac.changed(CampaignWithCustomDayBudget.DAY_BUDGET)) {
            BigDecimal newDayBudget = ac.getNewValue(CampaignWithCustomDayBudget.DAY_BUDGET);
            BigDecimal oldDayBudget = ac.getOldValue(CampaignWithCustomDayBudget.DAY_BUDGET);
            return ((isZero(newDayBudget) && greaterThanZero(oldDayBudget)) ||
                    (greaterThanZero(newDayBudget) && greaterThanZero(oldDayBudget)
                            && newDayBudget.compareTo(oldDayBudget) > 0));
        }
        return false;
    }

    @Override
    public void addToAdditionalActionsContainer(CampaignAdditionalActionsContainer additionalActionsContainer,
                                                RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                List<AppliedChanges<CampaignWithCustomDayBudget>> appliedChanges) {
        //noinspection ConstantConditions
        List<CampaignEvent<String>> campaignChangedDayBudgetEvents = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(CampaignWithCustomDayBudget.DAY_BUDGET))
                .map(ac -> CampaignEvent.changedDayBudgetEvent(updateParameters.getOperatorUid(),
                        updateParameters.getChiefUid(),
                        ac.getModel().getId(), ac.getOldValue(CampaignWithCustomDayBudget.CURRENCY),
                        ac.getOldValue(CampaignWithCustomDayBudget.DAY_BUDGET),
                        ac.getNewValue(CampaignWithCustomDayBudget.DAY_BUDGET)))
                .toList();

        additionalActionsContainer.addMailEventsToQueue(campaignChangedDayBudgetEvents);
    }

    @Override
    public void updateRelatedEntitiesInTransaction(DSLContext dslContext,
                                                   RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                   List<AppliedChanges<CampaignWithCustomDayBudget>> appliedChanges) {
        Collection<Long> campaignIdsToUpdate = filterAndMapList(appliedChanges,
                CampaignWithCustomDayBudgetUpdateOperationSupport::needUpdateStatusAutoBudgetShow,
                changes -> changes.getModel().getId());

        adGroupRepository.updateStatusAutoBudgetShowForCampaign(dslContext, campaignIdsToUpdate,
                StatusAutobudgetShow.YES);
    }

    private static boolean needUpdateStatusAutoBudgetShow(AppliedChanges<CampaignWithCustomDayBudget> changes) {
        /*
        сбрасываем установленные ранее отметки о приостановке показов автобюджетом если включили дневной бюджет
        или переключили режим показа
        */
        return greaterThanZero(changes.getModel().getDayBudget())
                && (isZero(changes.getOldValue(CampaignWithCustomDayBudget.DAY_BUDGET))
                || changes.changed(CampaignWithCustomDayBudget.DAY_BUDGET_SHOW_MODE));
    }

}
