package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.DayBudgetChangeLogRecord;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategyAndCustomDayBudget;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyService;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.strategySupportsDayBudget;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithCustomStrategyAndCustomDayBudgetUpdateOperationSupport
        extends AbstractCampaignUpdateOperationSupport<CampaignWithCustomStrategyAndCustomDayBudget> {

    private static final String LOGGER_NAME = "PPCLOG_CMD.log";
    private static final Logger changeLogger = LoggerFactory.getLogger(LOGGER_NAME);

    private final FeatureService featureService;

    @Autowired
    public CampaignWithCustomStrategyAndCustomDayBudgetUpdateOperationSupport(FeatureService featureService) {
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithCustomStrategyAndCustomDayBudget> getTypeClass() {
        return CampaignWithCustomStrategyAndCustomDayBudget.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                 List<AppliedChanges<CampaignWithCustomStrategyAndCustomDayBudget>> appliedChanges) {
        appliedChanges.forEach(changes -> {
            if (changes.changed(CampaignWithCustomStrategyAndCustomDayBudget.STRATEGY)
                    && !changes.changed(CampaignWithCustomStrategyAndCustomDayBudget.DAY_BUDGET)
                    && !BigDecimal.ZERO.equals(changes.getOldValue(
                            CampaignWithCustomStrategyAndCustomDayBudget.DAY_BUDGET))
                    && !strategySupportsDayBudget(
                    changes.getNewValue(CampaignWithCustomStrategyAndCustomDayBudget.STRATEGY))) {
                changes.modify(CampaignWithCustomStrategyAndCustomDayBudget.DAY_BUDGET, BigDecimal.ZERO);
                changes.modify(CampaignWithCustomStrategyAndCustomDayBudget.DAY_BUDGET_SHOW_MODE,
                        DayBudgetShowMode.DEFAULT_);
            }
        });
    }

    @Override
    public void updateRelatedEntitiesOutOfTransaction(
            RestrictedCampaignsUpdateOperationContainer updateParameters,
            List<AppliedChanges<CampaignWithCustomStrategyAndCustomDayBudget>> appliedChanges) {
        logDayBudgetChange(
                filterList(appliedChanges, ac -> ac.changed(CampaignWithCustomStrategyAndCustomDayBudget.DAY_BUDGET)),
                updateParameters.getOperatorUid());
    }

    void logDayBudgetChange(List<AppliedChanges<CampaignWithCustomStrategyAndCustomDayBudget>> changes,
                            Long operatorUid) {
        changes.stream()
                .map(ac -> getDayBudgetLogRecord(ac, operatorUid))
                .forEach(logRecord -> changeLogger.info("{} {}", logRecord.getPrefixLogTime(),
                        JsonUtils.toJson(logRecord)));
    }

    static DayBudgetChangeLogRecord getDayBudgetLogRecord(
            AppliedChanges<CampaignWithCustomStrategyAndCustomDayBudget> changes, Long operatorUid) {
        Long campaignId = changes.getModel().getId();
        DayBudgetChangeLogRecord dayBudgetChangeLogRecord = new DayBudgetChangeLogRecord(Trace.current().getTraceId());
        dayBudgetChangeLogRecord.setCids(List.of(campaignId));
        dayBudgetChangeLogRecord.setPath(DayBudgetChangeLogRecord.LOG_DAY_BUDGET_CHANGE_CMD_NAME);
        dayBudgetChangeLogRecord.setOperatorId(operatorUid);

        BigDecimal oldDayBudget = changes.getOldValue(CampaignWithCustomStrategyAndCustomDayBudget.DAY_BUDGET);
        BigDecimal newDayBudget = changes.getNewValue(CampaignWithCustomStrategyAndCustomDayBudget.DAY_BUDGET);
        CurrencyCode currencyCode = changes.getOldValue(CampaignWithCustomStrategyAndCustomDayBudget.CURRENCY);

        Map<String, Object> params = dayBudgetChangeLogRecord.getParam();
        params.put("cid", campaignId);
        params.put("old_day_budget", Objects.requireNonNull(oldDayBudget).toPlainString());
        params.put("new_day_budget", Objects.requireNonNull(newDayBudget).toPlainString());
        params.put("currency", Objects.requireNonNull(currencyCode).name());
        return dayBudgetChangeLogRecord;
    }

    @Override
    public void beforeExecution(
            RestrictedCampaignsUpdateOperationContainer updateContainer,
            List<AppliedChanges<CampaignWithCustomStrategyAndCustomDayBudget>> appliedChanges) {

        if (featureService.isEnabledForClientId(updateContainer.getClientId(), FeatureName.CPV_PAY_FOR_TRUE_VIEW)) {
            appliedChanges.forEach(changes -> {
                DbStrategy newStrategy = CampaignStrategyService
                        .correctCampaignStrategyUsePayForConversionValue(changes.getModel().getStrategy());
                changes.modify(CampaignWithCustomStrategyAndCustomDayBudget.STRATEGY, newStrategy);
            });
        }
    }
}
