package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter;
import ru.yandex.direct.core.entity.campaign.repository.CampMetrikaCountersRepository;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CommonCampaignCountersService;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toMetrikaCounters;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithMetrikaCountersUpdateOperationSupport extends AbstractCampaignUpdateOperationSupport<CampaignWithMetrikaCounters> {
    private final CampMetrikaCountersRepository campMetrikaCountersRepository;
    private final CommonCampaignCountersService commonCampaignCountersService;

    @Autowired
    public CampaignWithMetrikaCountersUpdateOperationSupport(CampMetrikaCountersRepository campMetrikaCountersRepository,
                                                             CommonCampaignCountersService commonCampaignCountersService) {
        this.campMetrikaCountersRepository = campMetrikaCountersRepository;
        this.commonCampaignCountersService = commonCampaignCountersService;
    }

    @Override
    public Class<CampaignWithMetrikaCounters> getTypeClass() {
        return CampaignWithMetrikaCounters.class;
    }

    @Override
    public void updateRelatedEntitiesInTransaction(DSLContext dslContext,
                                                   RestrictedCampaignsUpdateOperationContainer updateContainer,
                                                   List<AppliedChanges<CampaignWithMetrikaCounters>> appliedChanges) {
        List<CampaignWithMetrikaCounters> campaignWithMetrikaCountersList = filterAndMapList(
                appliedChanges,
                ac -> ac.changed(CampaignWithMetrikaCounters.METRIKA_COUNTERS),
                AppliedChanges::getModel);

        List<Long> metrikaCounterCidsToDelete = filterAndMapList(campaignWithMetrikaCountersList,
                campaign -> CollectionUtils.isEmpty(campaign.getMetrikaCounters()),
                CampaignWithMetrikaCounters::getId);

        campMetrikaCountersRepository.deleteMetrikaCounters(dslContext, metrikaCounterCidsToDelete);

        List<CampaignWithMetrikaCounters> campaignWithMetrikaCountersToUpdate =
                filterList(campaignWithMetrikaCountersList,
                        campaign -> CollectionUtils.isNotEmpty(campaign.getMetrikaCounters()));

        if (campaignWithMetrikaCountersToUpdate.isEmpty()) {
            return;
        }

        var counterIdsByCampaignId = listToMap(
                campaignWithMetrikaCountersToUpdate,
                CampaignWithMetrikaCounters::getId,
                campaign -> listToSet(campaign.getMetrikaCounters())
        );

        var availableCounters = commonCampaignCountersService.getRelevantCounters(
                updateContainer.getClientId(),
                counterIdsByCampaignId,
                updateContainer.getMetrikaClient().getUsersCountersNumExtendedByCampaignCounterIds(),
                updateContainer.getMetrikaClient().getCampaignCountersIfNeeded()
        );

        var availableCountersById = StreamEx.of(availableCounters.values())
                .flatCollection(Function.identity())
                .distinct(MetrikaCounterWithAdditionalInformation::getId)
                .toMap(MetrikaCounterWithAdditionalInformation::getId, Function.identity());

        Map<Long, List<MetrikaCounter>> countersByCid =
                StreamEx.of(appliedChanges)
                        .filter(x -> x.changed(CampaignWithMetrikaCounters.METRIKA_COUNTERS))
                        .map(AppliedChanges::getModel)
                        .mapToEntry(CampaignWithMetrikaCounters::getId, CampaignWithMetrikaCounters::getMetrikaCounters)
                        .nonNullValues()
                        .mapValues(counters -> toMetrikaCounters(counters, availableCountersById))
                        .toMap();

        campMetrikaCountersRepository.updateMetrikaCounters(dslContext, countersByCid);
    }

    @Override
    public void addToAdditionalActionsContainer(CampaignAdditionalActionsContainer additionalActionsContainer,
                                                RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                List<AppliedChanges<CampaignWithMetrikaCounters>> appliedChanges) {
        Set<Long> campaignIdsWithChangedMetrikaCounters = filterAndMapToSet(appliedChanges,
                ac -> ac.collectionContentChanged(CampaignWithMetrikaCounters.METRIKA_COUNTERS),
                ac -> ac.getModel().getId());

        additionalActionsContainer.resetCampaignBannersStatusBsSynced(campaignIdsWithChangedMetrikaCounters);
    }
}
