package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithOrganizationAndPhone;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrganizationAndPhone;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.DefectId;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithOrganizationAndPhoneUpdateOperationSupport
        extends AbstractCampaignUpdateOperationSupport<CampaignWithOrganizationAndPhone> {

    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithOrganizationAndPhoneUpdateOperationSupport.class);

    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final FeatureService featureService;

    @Autowired
    public CampaignWithOrganizationAndPhoneUpdateOperationSupport(BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                                                  BannerRelationsRepository bannerRelationsRepository,
                                                                  FeatureService featureService) {
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithOrganizationAndPhone> getTypeClass() {
        return CampaignWithOrganizationAndPhone.class;
    }

    @Override
    public void beforeExecution(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                List<AppliedChanges<CampaignWithOrganizationAndPhone>> appliedChanges) {
        // Удаляем телефон при удалении организации
        appliedChanges.forEach(ac -> {
            if (ac.getNewValue(CampaignWithOrganizationAndPhone.DEFAULT_CHAIN_ID) == null &&
                    ac.getNewValue(CampaignWithOrganizationAndPhone.DEFAULT_PERMALINK_ID) == null) {
                ac.modify(CampaignWithOrganizationAndPhone.DEFAULT_TRACKING_PHONE_ID, null);
            }
        });
    }

    @Override
    public void updateRelatedEntitiesOutOfTransaction(
            RestrictedCampaignsUpdateOperationContainer updateContainer,
            List<AppliedChanges<CampaignWithOrganizationAndPhone>> appliedChanges) {
        if (featureService.isEnabledForClientId(updateContainer.getClientId(),
                FeatureName.CHANGE_BANNER_ORGANIZATION_ON_DEFAULT_CAMPAIGN_ORGANIZATION_CHANGE)) {

            var campaignsWithChangedOrganizationOrPhone = filterList(appliedChanges,
                    ac -> ac.changed(CampaignWithOrganizationAndPhone.DEFAULT_PERMALINK_ID) ||
                            ac.changed(CampaignWithOrganizationAndPhone.DEFAULT_TRACKING_PHONE_ID));

            updateBannerOrganizations(updateContainer, campaignsWithChangedOrganizationOrPhone);
        }
    }

    private void updateBannerOrganizations(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                           List<AppliedChanges<CampaignWithOrganizationAndPhone>> campaigns) {
        if (campaigns.isEmpty()) {
            return;
        }

        var campaignIds = mapList(campaigns, campaign -> campaign.getModel().getId());

        var campaignsById = listToMap(campaigns,
                campaign -> campaign.getModel().getId(), AppliedChanges::getModel);

        var bannerIds =
                bannerRelationsRepository.getBannerIdsMapByCampaignIdsAndBannerTypes(updateContainer.getShard(),
                        campaignIds, Collections.singleton(BannerWithOrganizationAndPhone.class));

        List<ModelChanges<BannerWithOrganizationAndPhone>> modelChanges = EntryStream.of(bannerIds)
                .mapValues(campaignsById::get)
                .mapKeyValue((bid, campaign) -> new ModelChanges<>(bid, BannerWithOrganizationAndPhone.class)
                        .process(campaign.getDefaultPermalinkId(), BannerWithOrganizationAndPhone.PERMALINK_ID)
                        .process(campaign.getDefaultTrackingPhoneId(), BannerWithOrganizationAndPhone.PHONE_ID)
                        .process(false, BannerWithOrganizationAndPhone.PREFER_V_CARD_OVER_PERMALINK))
                .toList();

        var updateOperation =
                bannersUpdateOperationFactory.createPartialUpdateOperation(modelChanges,
                        updateContainer.getOperatorUid(), updateContainer.getClientId(),
                        BannerWithOrganizationAndPhone.class);

        var result = updateOperation.prepareAndApply();
        logger.info("Result of updating organizations for {} banners: {} OK, {} ERROR due to defects {}",
                modelChanges.size(),
                result.getSuccessfulCount(),
                result.getErrorCount(),
                getDefects(result));
    }

    private Set<DefectId> getDefects(MassResult<Long> result) {
        return listToSet(result.getValidationResult().flattenErrors(), defectInfo -> defectInfo.getDefect().defectId());
    }
}
