package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignAdditionalActionsService;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainerService;
import ru.yandex.direct.core.entity.strategy.container.StrategyOperationOptions;
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainerService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository;
import ru.yandex.direct.core.entity.strategy.service.add.StrategyAddOperationService;
import ru.yandex.direct.core.entity.strategy.service.converter.CampaignToStrategyConverterService;
import ru.yandex.direct.core.entity.strategy.service.converter.StrategyToCampaignConverterFacade;
import ru.yandex.direct.core.entity.strategy.service.update.StrategyUpdateOperationService;
import ru.yandex.direct.core.entity.strategy.type.common.CommonStrategyUpdateLastChangeService;
import ru.yandex.direct.core.entity.strategy.utils.StrategyComparingUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.PROPERTIES_BY_TYPE;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.STRATEGY_TYPE_TO_CLASS;
import static ru.yandex.direct.feature.FeatureName.PACKAGE_STRATEGIES_STAGE_TWO;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyUpdateOperationSupport
        extends AbstractCampaignWithPackageUpdateOperationSupport {

    private static final Set<ModelProperty<? super CommonStrategy, ?>> STRATEGY_PROPERTIES_TO_IGNORE_WHEN_UPDATE_FROM_CAMPAIGN = Set.of(
            CommonStrategy.NAME,
            CommonStrategy.CIDS,
            CommonStrategy.WALLET_ID,
            CommonStrategy.STATUS_ARCHIVED);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final FeatureService featureService;
    private final DslContextProvider dslContextProvider;
    private final CampaignToStrategyConverterService campaignToStrategyConverterService;
    private final StrategyToCampaignConverterFacade strategyToCampaignConverterFacade;
    private final StrategyAddOperationService strategyAddOperationService;
    private final StrategyUpdateOperationService strategyUpdateOperationService;
    private final StrategyUpdateOperationContainerService strategyUpdateOperationContainerService;
    private final StrategyAddOperationContainerService strategyAddOperationContainerService;
    private final CampaignAdditionalActionsService campaignAdditionalActionsService;

    private final StrategyTypedRepository strategyTypedRepository;
    private final CampaignTypedRepository campaignTypedRepository;

    @Autowired
    public CampaignWithPackageStrategyUpdateOperationSupport(
            PpcPropertiesSupport ppcPropertiesSupport,
            FeatureService featureService,
            DslContextProvider dslContextProvider,
            CampaignToStrategyConverterService campaignToStrategyConverterService,
            StrategyToCampaignConverterFacade strategyToCampaignConverterFacade,
            StrategyAddOperationService strategyAddOperationService,
            StrategyUpdateOperationService strategyUpdateOperationService,
            StrategyUpdateOperationContainerService strategyUpdateOperationContainerService,
            StrategyAddOperationContainerService strategyAddOperationContainerService,
            CampaignAdditionalActionsService campaignAdditionalActionsService,
            StrategyTypedRepository strategyTypedRepository,
            CampaignTypedRepository campaignTypedRepository) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.featureService = featureService;
        this.dslContextProvider = dslContextProvider;
        this.campaignToStrategyConverterService = campaignToStrategyConverterService;
        this.strategyToCampaignConverterFacade = strategyToCampaignConverterFacade;
        this.strategyAddOperationService = strategyAddOperationService;
        this.strategyUpdateOperationService = strategyUpdateOperationService;
        this.strategyUpdateOperationContainerService = strategyUpdateOperationContainerService;
        this.strategyAddOperationContainerService = strategyAddOperationContainerService;
        this.campaignAdditionalActionsService = campaignAdditionalActionsService;
        this.strategyTypedRepository = strategyTypedRepository;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    @Override
    public void onModelChangesValidated(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                        List<ModelChanges<CampaignWithPackageStrategy>> validModelChanges) {
        var oldCampaignById = campaignTypedRepository.getTypedCampaignsMap(
                updateContainer.getShard(),
                mapList(validModelChanges, ModelChanges::getId));

        boolean isPackageStrategiesStageTwoFeatureEnabled =
                featureService.isEnabledForClientId(updateContainer.getClientId(), PACKAGE_STRATEGIES_STAGE_TWO);

        var mcWithChangedStrategyId = filterList(validModelChanges,
                mc -> mc.isPropChanged(CampaignWithPackageStrategy.STRATEGY_ID));

        if (!isPackageStrategiesStageTwoFeatureEnabled) {
            // если фича второго этапа не включена, то и изменения должны происходить только в стратегии кампании
            mcWithChangedStrategyId
                    .forEach(mc -> mc.process(
                            ((CampaignWithPackageStrategy) oldCampaignById.get(mc.getId())).getStrategyId(),
                            CampaignWithPackageStrategy.STRATEGY_ID
                    ));
            return;
        }

        Map<Long, BaseStrategy> packageStrategyByCidWithChangedStrategyId =
                getPackageStrategyByCidWithChangedStrategyId(updateContainer, mcWithChangedStrategyId);

        LocalDateTime now = now();
        mcWithChangedStrategyId.forEach(mc -> updateModelChanges(
                mc, (CampaignWithPackageStrategy) oldCampaignById.get(mc.getId()),
                Map.copyOf(packageStrategyByCidWithChangedStrategyId), now));
    }

    @Override
    public void updateRelatedEntitiesInTransaction(
            DSLContext dslContext,
            RestrictedCampaignsUpdateOperationContainer updateContainer,
            List<AppliedChanges<CampaignWithPackageStrategy>> appliedChanges) {
        boolean isPackageStrategiesStageTwoFeatureEnabled =
                featureService.isEnabledForClientId(updateContainer.getClientId(), PACKAGE_STRATEGIES_STAGE_TWO);

        var oldCampaignStrategyById = getOldCampaignStrategyById(dslContext, appliedChanges);
        var isAnyImportantForPackageStrategyPropChangedByCid =
                isAnyImportantForPackageStrategyPropChangedByCid(updateContainer, appliedChanges,
                        oldCampaignStrategyById);

        if (isPackageStrategiesStageTwoFeatureEnabled) {
            createPackageStrategiesForCampaignsIfNecessary(dslContext, updateContainer, appliedChanges,
                    oldCampaignStrategyById, isAnyImportantForPackageStrategyPropChangedByCid);
        }

        if (!updateContainer.getOptions().isFromStrategyOperation()) {
            updateRelatedPackageStrategies(dslContext, updateContainer, appliedChanges,
                    isPackageStrategiesStageTwoFeatureEnabled, isAnyImportantForPackageStrategyPropChangedByCid);
        }
    }

    @Override
    public void onAppliedChangesValidated(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                          List<AppliedChanges<CampaignWithPackageStrategy>> appliedChanges) {
        CampaignAdditionalActionsContainer campaignAdditionalActionsContainer =
                new CampaignAdditionalActionsContainer();
        Set<Long> cidsWithChangedStrategyId = StreamEx.of(appliedChanges)
                .filter(changes -> changes.changed(CampaignWithPackageStrategy.STRATEGY_ID))
                .map(changes -> changes.getModel().getId())
                .toSet();
        campaignAdditionalActionsContainer.resetCampaignBannersStatusBsSynced(cidsWithChangedStrategyId);
        DSLContext dslContext = dslContextProvider.ppc(updateContainer.getShard());
        campaignAdditionalActionsService.resetBannersBsStatusSynced(dslContext, campaignAdditionalActionsContainer);
    }

    private void updateModelChanges(ModelChanges<CampaignWithPackageStrategy> mc,
                                    CampaignWithPackageStrategy oldCampaign,
                                    Map<Long, BaseStrategy> packageStrategyByCidWithChangedStrategyId,
                                    LocalDateTime now) {
        var newStrategyId = mc.getChangedProp(CampaignWithPackageStrategy.STRATEGY_ID);

        boolean isMcWithDifferentStrategyId = !Objects.equals(oldCampaign.getStrategyId(), newStrategyId);
        boolean shouldCreateNewPackageStrategyWhenReset = isMcWithDifferentStrategyId && newStrategyId == null;
        boolean shouldMoveToPackageFromStrategyId = isMcWithDifferentStrategyId && newStrategyId != null;

        if (shouldCreateNewPackageStrategyWhenReset) {
            // в базе поле strategyId не nullable, поэтому пока запишем 0,
            // а в методе updateRelatedEntitiesInTransaction создадим новую непубличную стратегию
            // и обновим strategyId на валидный
            mc.process(0L, CampaignWithPackageStrategy.STRATEGY_ID);
        } else if (shouldMoveToPackageFromStrategyId) {
            var newStrategy = packageStrategyByCidWithChangedStrategyId.get(mc.getId());
            var campOptionStrategy = mc.isPropChanged(CampaignWithPackageStrategy.STRATEGY) ?
                    mc.getChangedProp(CampaignWithPackageStrategy.STRATEGY).getStrategy() :
                    oldCampaign.getStrategy().getStrategy();

            var platform = mc.isPropChanged(CampaignWithPackageStrategy.STRATEGY) ?
                    mc.getChangedProp(CampaignWithPackageStrategy.STRATEGY).getPlatform() :
                    oldCampaign.getStrategy().getPlatform();

            strategyToCampaignConverterFacade.copyStrategyToCampaignModelChanges(
                    now, newStrategy, mc, oldCampaign.getClass());

            DbStrategy strategy = mc.getChangedProp(CampaignWithPackageStrategy.STRATEGY);
            strategy.setStrategy(campOptionStrategy);
            strategy.setPlatform(platform);
        }
    }

    private Map<Long, BaseStrategy> getOldCampaignStrategyById(DSLContext dslContext,
                                                               List<AppliedChanges<CampaignWithPackageStrategy>> appliedChanges) {
        var filledStrategyIds = filterAndMapList(
                appliedChanges,
                ac -> {
                    var oldStrategyId = ac.getOldValue(CampaignWithPackageStrategy.STRATEGY_ID);
                    return oldStrategyId != null && oldStrategyId != 0;
                },
                ac -> ac.getOldValue(CampaignWithPackageStrategy.STRATEGY_ID));
        return strategyTypedRepository.getIdToModelTyped(dslContext, filledStrategyIds);
    }

    private Map<Long, Boolean> isAnyImportantForPackageStrategyPropChangedByCid(
            RestrictedCampaignsUpdateOperationContainer updateContainer,
            List<AppliedChanges<CampaignWithPackageStrategy>> appliedChanges,
            Map<Long, BaseStrategy> oldCampaignStrategyByStrategyId) {
        LocalDateTime now = now();
        return StreamEx.of(appliedChanges)
                .mapToEntry(ac -> ac.getModel().getId(), ac -> {
                    var oldStrategyId = ac.getOldValue(CampaignWithPackageStrategy.STRATEGY_ID);
                    if (oldStrategyId == 0) {
                        return false;
                    }
                    var oldStrategy = oldCampaignStrategyByStrategyId.get(oldStrategyId);
                    var newStrategy = campaignToStrategyConverterService.toStrategyWithIdEqualToCampaignStrategyId(
                            updateContainer.getClientId(), now, ac.getModel());
                    return StrategyComparingUtils.areDifferentStrategies(oldStrategy, newStrategy);
                }).toMap();
    }

    private Map<Long, BaseStrategy> getPackageStrategyByCidWithChangedStrategyId(RestrictedCampaignsUpdateOperationContainer updateOperationContainer,
                                                                                 List<ModelChanges<CampaignWithPackageStrategy>> modelChangesWithChangedStrategyId) {
        var filledChangedStrategyIdByCid = StreamEx.of(modelChangesWithChangedStrategyId)
                .mapToEntry(ModelChanges::getId, mc -> mc.getChangedProp(CampaignWithPackageStrategy.STRATEGY_ID))
                .nonNullValues()
                .toMap();

        Map<Long, BaseStrategy> newStrategyById =
                strategyTypedRepository.getIdToModelTyped(updateOperationContainer.getShard(),
                        Set.copyOf(filledChangedStrategyIdByCid.values()));

        return EntryStream.of(filledChangedStrategyIdByCid)
                .mapValues(newStrategyById::get)
                .toMap();
    }

    private void createPackageStrategiesForCampaignsIfNecessary(DSLContext dslContext,
                                                                RestrictedCampaignsUpdateOperationContainer updateContainer,
                                                                List<AppliedChanges<CampaignWithPackageStrategy>> appliedChanges,
                                                                Map<Long, BaseStrategy> oldCampaignStrategyByStrategyId,
                                                                Map<Long, Boolean> isAnyImportantForPackageStrategyPropChangedByCid) {
        var campaignsToCreatePackageStrategy = StreamEx.of(appliedChanges)
                .filter(ac -> shouldCreateNewPackageStrategy(ac, oldCampaignStrategyByStrategyId,
                        isAnyImportantForPackageStrategyPropChangedByCid))
                .map(AppliedChanges::getModel)
                .toList();

        createPackageStrategiesForCampaigns(updateContainer, dslContext, campaignsToCreatePackageStrategy,
                appliedChanges);
    }

    private boolean shouldCreateNewPackageStrategy(AppliedChanges<CampaignWithPackageStrategy> ac,
                                                   Map<Long, BaseStrategy> oldCampaignStrategyByStrategyId,
                                                   Map<Long, Boolean> isAnyImportantForPackageStrategyPropChangedByCid) {
        Long curStrategyId = ac.getModel().getStrategyId();
        Long oldStrategyId = ac.getOldValue(CampaignWithPackageStrategy.STRATEGY_ID);

        var oldStrategy = (CommonStrategy) oldCampaignStrategyByStrategyId.get(oldStrategyId);
        return oldStrategy.getIsPublic()
                && (!ac.changed(CampaignWithPackageStrategy.STRATEGY_ID)
                && isAnyImportantForPackageStrategyPropChangedByCid.get(ac.getModel().getId())
                || curStrategyId == 0);
    }

    private void createPackageStrategiesForCampaigns(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                                     DSLContext dslContext,
                                                     List<CampaignWithPackageStrategy> campaignsToCreatePackageStrategy,
                                                     List<AppliedChanges<CampaignWithPackageStrategy>> appliedChanges) {
        LocalDateTime now = now();
        var strategiesToAddByCampaignId = listToMap(
                campaignsToCreatePackageStrategy,
                BaseCampaign::getId,
                campaign -> campaignToStrategyConverterService.toStrategyWithNotFilledId(
                        updateContainer.getClientId(), now, campaign));
        StrategyOperationOptions options =
                new StrategyOperationOptions(
                        false,
                        false,
                        false,
                        false,
                        false,
                        false,
                        true,
                        ppcPropertiesSupport.get(PpcPropertyNames.MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY)
                                .getOrDefault(DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY));
        StrategyAddOperationContainer container = new StrategyAddOperationContainer(updateContainer.getShard(),
                updateContainer.getClientId(), updateContainer.getClientUid(),
                updateContainer.getOperatorUid(), options);
        List<BaseStrategy> strategies = List.copyOf(strategiesToAddByCampaignId.values());
        strategyAddOperationContainerService.fillContainers(container, strategies, dslContext);
        strategyAddOperationService.execute(dslContext, container, strategies);
        var cidsWithCreatedPackageStrategies = mapList(campaignsToCreatePackageStrategy,
                CampaignWithPackageStrategy::getId);
        appliedChanges.forEach(ac -> {
            if (cidsWithCreatedPackageStrategies.contains(ac.getModel().getId())) {
                ac.modify(CampaignWithPackageStrategy.STRATEGY_ID,
                        strategiesToAddByCampaignId.get(ac.getModel().getId()).getId());
            }
        });
    }

    private void updateRelatedPackageStrategies(DSLContext dslContext,
                                                RestrictedCampaignsUpdateOperationContainer updateContainer,
                                                List<AppliedChanges<CampaignWithPackageStrategy>> appliedChanges,
                                                boolean isPackageStrategiesStageTwoFeatureEnabled,
                                                Map<Long, Boolean> isAnyImportantForPackageStrategyPropChangedByCid) {
        LocalDateTime now = now();
        var campaignChangesToUpdateStrategy =
                filterList(appliedChanges,
                        ac -> {
                            Long strategyId = ac.getModel().getStrategyId();
                            return strategyId != 0
                                    && !ac.changed(CampaignWithPackageStrategy.STRATEGY_ID)
                                    && isAnyImportantForPackageStrategyPropChangedByCid.get(ac.getModel().getId());
                        });

        var strategiesModelChanges = calculateStrategiesModelChanges(
                now,
                updateContainer,
                campaignChangesToUpdateStrategy,
                isPackageStrategiesStageTwoFeatureEnabled);

        List<Long> strategiesToUpdateIds = mapList(campaignChangesToUpdateStrategy,
                ac -> ac.getModel().getStrategyId());

        if (isPackageStrategiesStageTwoFeatureEnabled) {
            updateModelChangesStrategiesToPublicIfNecessary(updateContainer, appliedChanges, strategiesModelChanges,
                    strategiesToUpdateIds, now);

            var strategyRelationChangedIds = CommonStrategyUpdateLastChangeService.Companion
                    .getOldStrategyIdsFromCampaignChanges(appliedChanges);

            var strategiesRelationChanges = strategyRelationChangedIds.stream()
                    .map(id -> ModelChanges.build(id, CommonStrategy.class, CommonStrategy.LAST_CHANGE, now))
                    .map(mc -> mc.castModel(BaseStrategy.class))
                    .collect(Collectors.toList());

            strategiesModelChanges.addAll(strategiesRelationChanges);
            strategiesToUpdateIds.addAll(strategyRelationChangedIds);
        }

        Map<Long, BaseStrategy> strategyById = strategyTypedRepository.getIdToModelTyped(dslContext,
                strategiesToUpdateIds);

        strategyUpdateOperationService.executeFromCampaign(dslContext,
                updateContainer,
                strategyById,
                strategiesModelChanges);
    }

    private List<ModelChanges<BaseStrategy>> calculateStrategiesModelChanges(
            LocalDateTime now,
            RestrictedCampaignsUpdateOperationContainer updateContainer,
            List<AppliedChanges<CampaignWithPackageStrategy>> changesToUpdateStrategy,
            boolean isPackageStrategiesStageTwoFeatureEnabled) {
        return mapList(changesToUpdateStrategy, ac -> {
            BaseStrategy newStrategy =
                    campaignToStrategyConverterService.toStrategyWithIdEqualToCampaignStrategyId(
                            updateContainer.getClientId(),
                            now,
                            ac.getModel());

            if (isPackageStrategiesStageTwoFeatureEnabled) {
                newStrategy.setId(ac.getModel().getStrategyId());
            }
            //noinspection unchecked
            return calculateStrategyModelChanges(newStrategy,
                    (Class<BaseStrategy>) STRATEGY_TYPE_TO_CLASS.get(newStrategy.getType()));
        });
    }

    private <T extends BaseStrategy> ModelChanges<T> calculateStrategyModelChanges(T strategy, Class<T> strategyClazz) {
        ModelChanges<T> modelChanges = new ModelChanges<>(strategy.getId(), strategyClazz);
//        noinspection unchecked
        StreamEx.of(PROPERTIES_BY_TYPE.get(strategy.getType()))
                .filter(property -> !STRATEGY_PROPERTIES_TO_IGNORE_WHEN_UPDATE_FROM_CAMPAIGN.contains(property))
                .forEach(property -> processStrategyProperty(strategy, modelChanges, (ModelProperty<T, ?>) property));
        return modelChanges;
    }

    private <T extends BaseStrategy, V> void processStrategyProperty(T strategy, ModelChanges<T> modelChanges,
                                                                     ModelProperty<T, V> property) {
        modelChanges.process(property.get(strategy), property);
    }

    private void updateModelChangesStrategiesToPublicIfNecessary(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                                                 List<AppliedChanges<CampaignWithPackageStrategy>> modifiedModels,
                                                                 List<ModelChanges<BaseStrategy>> strategiesModelChanges,
                                                                 List<Long> strategiesToUpdateIds,
                                                                 LocalDateTime now) {
        Set<Long> strategiesToBecomePublicIds = StreamEx.of(modifiedModels)
                .map(campaign -> (CommonStrategy) updateContainer.getPackageStrategy(campaign.getModel().getStrategyId()))
                .filter(Objects::nonNull) // отфильтровываем стратегии, созданные в операции (их нет в контейнере)
                .filter(strategy -> shouldBecomePublic(strategy, modifiedModels))
                .map(BaseStrategy::getId)
                .toImmutableSet();

        strategiesModelChanges.forEach(
                mc -> {
                    if (strategiesToBecomePublicIds.contains(mc.getId())) {
                        mc.castModel(CommonStrategy.class)
                                .process(true, CommonStrategy.IS_PUBLIC)
                                .process(String.format("Strategy %d dated %s", mc.getId(), now.toLocalDate()),
                                        CommonStrategy.NAME);
                        strategiesToBecomePublicIds.remove(mc.getId());
                    }
                }
        );

        strategiesToBecomePublicIds.forEach(
                strategyId -> strategiesModelChanges.add(
                        new ModelChanges<>(strategyId, CommonStrategy.class)
                                .process(true, CommonStrategy.IS_PUBLIC)
                                .process(String.format("Strategy %d dated %s", strategyId, now.toLocalDate()),
                                        CommonStrategy.NAME)
                                .castModel(BaseStrategy.class))
        );
        strategiesToUpdateIds.addAll(strategiesToBecomePublicIds);
    }

    private static boolean shouldBecomePublic(CommonStrategy strategy,
                                              List<AppliedChanges<CampaignWithPackageStrategy>> modifiedModels) {
        Boolean isPublic = strategy.getIsPublic();
        boolean isNotPublicStrategy = isPublic == null || !isPublic;

        boolean willHaveNewLinkedCampaigns = filterList(modifiedModels,
                ac -> ac.getModel().getStrategyId().equals(strategy.getId())
                        && ac.changed(CampaignWithPackageStrategy.STRATEGY_ID)
        ).size() != 0;

        return isNotPublicStrategy && willHaveNewLinkedCampaigns;
    }

    @Override
    public Class<CampaignWithPackageStrategy> getTypeClass() {
        return CampaignWithPackageStrategy.class;
    }

}
