package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.service.pricerecalculation.CpmPriceCampaignPriceRecalculationService;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.enrichBidModifierInventory;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.getStrategy;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.pricePackageHasImpressionRate;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPricePackageUpdateOperationSupport
        extends AbstractCampaignUpdateOperationSupport<CampaignWithPricePackage> {

    private final PricePackageRepository pricePackageRepository;
    private final CpmPriceCampaignPriceRecalculationService cpmPriceCampaignPriceRecalculationService;
    private final FeatureService featureService;


    @Autowired
    public CampaignWithPricePackageUpdateOperationSupport(PricePackageRepository pricePackageRepository,
                                                          CpmPriceCampaignPriceRecalculationService
                                                                      cpmPriceCampaignPriceRecalculationService,
                                                          FeatureService featureService) {
        this.pricePackageRepository = pricePackageRepository;
        this.cpmPriceCampaignPriceRecalculationService = cpmPriceCampaignPriceRecalculationService;
        this.featureService = featureService;

    }

    @Override
    public Class<CampaignWithPricePackage> getTypeClass() {
        return CampaignWithPricePackage.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer container,
                                 List<AppliedChanges<CampaignWithPricePackage>> appliedChanges) {
        Map<Long, PricePackage> pricePackages = getPricePackages(appliedChanges);
        var calcBudgetFeature = featureService.isEnabledForClientId(container.getClientId(),
                FeatureName.BACKEND_CPM_PRICE_CAMPAIGN_BUDGET_CALC_ENABLED);

        for (AppliedChanges<CampaignWithPricePackage> appliedChange : appliedChanges) {
            CampaignWithPricePackage campaign = appliedChange.getModel();
            PricePackage pricePackage = pricePackages.get(campaign.getPricePackageId());

            appliedChange.modify(CampaignWithPricePackage.STRATEGY,
                    getStrategy(campaign, pricePackage, calcBudgetFeature));
            if (pricePackageHasImpressionRate(pricePackage)
                    && appliedChange.deleted(CampaignWithPricePackage.IMPRESSION_RATE_COUNT)) {
                //Если в пакете явно указана частота, она копируется на РК.
                // Изменения на null в appliedChange трактуем как без изменений
                var packageFrequencyLimit = pricePackage.getCampaignOptions().getShowsFrequencyLimit();
                appliedChange.modify(CampaignWithPricePackage.IMPRESSION_RATE_COUNT,
                        packageFrequencyLimit.getFrequencyLimit());
                appliedChange.modify(CampaignWithPricePackage.IMPRESSION_RATE_INTERVAL_DAYS,
                        packageFrequencyLimit.getFrequencyLimitDays());
            }
            if (appliedChange.changed(CampaignWithPricePackage.BID_MODIFIERS)) {
                //Для прайсовой кампании редактируются опциональные корректировки на кампанию,
                // обязательные находятся на пакете.

                appliedChange.modify(CampaignWithPricePackage.BID_MODIFIERS,
                        enrichBidModifierInventory(pricePackage, campaign));
            }
        }
    }

    @Override
    public void beforeExecution(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                List<AppliedChanges<CampaignWithPricePackage>> appliedChanges) {
        Map<Long, PricePackage> pricePackages = getPricePackages(appliedChanges);

        appliedChanges.forEach(ac -> {
            CampaignWithPricePackage campaign = ac.getModel();
            PricePackage pricePackage = pricePackages.get(campaign.getPricePackageId());
            if (ac.changed(CampaignWithPricePackage.START_DATE)
                    || ac.changed(CampaignWithPricePackage.END_DATE)
                    || ac.changed(CampaignWithPricePackage.FLIGHT_ORDER_VOLUME)) {
                // для любого типа прайсового продукта (в не зависимости от разрешенных availableAdGroupTypes )
                // если на продукте стоит campaignAutoApprove = true
                // то status_approve на кампании не сбрасываем, иначе сбрасываем.
                if (pricePackage.getCampaignAutoApprove() == null || !pricePackage.getCampaignAutoApprove()) {
                    ac.modify(CampaignWithPricePackage.FLIGHT_STATUS_APPROVE, PriceFlightStatusApprove.NEW);
                }
            }
            if ((ac.changed(CampaignWithPricePackage.START_DATE) || ac.changed(CampaignWithPricePackage.END_DATE)) &&
                    pricePackage.isFrontpagePackage() &&
                    cpmPriceCampaignPriceRecalculationService.needRejectCampaignApprove(pricePackage, ac.getModel())) {
                // Если изменились даты начала или конца кампании, надо пересчитать сезонные коэф,
                // а там возможен вариант, что надо отклонить апрув кампании
                ac.modify(CampaignWithPricePackage.FLIGHT_STATUS_APPROVE, PriceFlightStatusApprove.NO);
            }
        });
    }

    @Override
    public void addToAdditionalActionsContainer(CampaignAdditionalActionsContainer additionalActionsContainer,
                                                RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                List<AppliedChanges<CampaignWithPricePackage>> appliedChanges) {
        Set<Long> campaignIdsForMarkAggregatedStatusesAsObsolete = filterAndMapToSet(appliedChanges,
                ac -> ac.changed(CampaignWithPricePackage.FLIGHT_STATUS_APPROVE),
                ac -> ac.getModel().getId());
        additionalActionsContainer
                .addCampaignIdsForMarkAggregatedStatusesAsObsolete(campaignIdsForMarkAggregatedStatusesAsObsolete);
    }

    private Map<Long, PricePackage> getPricePackages(List<AppliedChanges<CampaignWithPricePackage>> appliedChanges) {
        Set<Long> pricePackageIds = appliedChanges.stream()
                .map(AppliedChanges::getModel)
                .map(CampaignWithPricePackage::getPricePackageId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        return pricePackageRepository.getPricePackages(pricePackageIds);
    }

    @Override
    public void updateRelatedEntitiesOutOfTransaction(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                                      List<AppliedChanges<CampaignWithPricePackage>> appliedChanges) {

        List<AppliedChanges<CampaignWithPricePackage>> campaignsChangesWithChangedStartDate =
                filterList(appliedChanges, ac -> ac.changed(CampaignWithPricePackage.START_DATE)
                || ac.changed(CampaignWithPricePackage.END_DATE));

        cpmPriceCampaignPriceRecalculationService.afterCpmPriceCampaignsDateChanged(
                campaignsChangesWithChangedStartDate,
                UidClientIdShard.of(updateContainer.getChiefUid(), updateContainer.getClientId(),
                        updateContainer.getShard()));
    }
}
