package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.MailTextCreatorService;
import ru.yandex.direct.core.entity.campaign.service.type.add.AddServicedCampaignService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.AddServicedCampaignInfo;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.mailnotification.model.CampaignEvent;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.AutoServicingMailNotification;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.common.util.RepositoryUtils.NOW_PLACEHOLDER;
import static ru.yandex.direct.utils.CommonUtils.isValidId;

@Component
@ParametersAreNonnullByDefault
public class CommonCampaignUpdateOperationSupport extends AbstractCampaignUpdateOperationSupport<CommonCampaign> {

    private final MailTextCreatorService mailTextCreatorService;
    private final AddServicedCampaignService addServicedCampaignService;
    private final ClientService clientService;
    private final CampaignService campaignService;
    private final UserService userService;
    private final NotificationService notificationService;

    @Autowired
    public CommonCampaignUpdateOperationSupport(MailTextCreatorService mailTextCreatorService,
                                                AddServicedCampaignService addServicedCampaignService,
                                                ClientService clientService,
                                                CampaignService campaignService,
                                                UserService userService,
                                                NotificationService notificationService) {
        this.mailTextCreatorService = mailTextCreatorService;
        this.addServicedCampaignService = addServicedCampaignService;
        this.clientService = clientService;
        this.campaignService = campaignService;
        this.userService = userService;
        this.notificationService = notificationService;
    }

    @Override
    public Class<CommonCampaign> getTypeClass() {
        return CommonCampaign.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer container,
                                 List<AppliedChanges<CommonCampaign>> appliedChanges) {
        AddServicedCampaignInfo servicedInfo = addServicedCampaignService.getServicedInfoForClientCampaigns(container);
        appliedChanges.forEach(changes -> {
            CommonCampaign campaign = changes.getModel();
            changes.modify(CommonCampaign.LAST_CHANGE, NOW_PLACEHOLDER);
            changes.modify(CommonCampaign.STATUS_BS_SYNCED, CampaignStatusBsSynced.NO);

            if (changes.changed(CommonCampaign.TIME_TARGET)) {
                changes.modify(CommonCampaign.AUTOBUDGET_FORECAST_DATE, null);
            }

            if (Boolean.TRUE.equals(container.getRequireServicing(campaign))
                    && servicedInfo.getManagerUid() != null
                    && campaign.getManagerUid() == null) {
                changes.modify(CommonCampaign.MANAGER_UID, servicedInfo.getManagerUid());

                // https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/CampaignTools.pm?rev=r9190755#L513
                // if not exist moderated banner and some banners on moderation now
                // small hack for change campaign's status
                if (campaign.getStatusPostModerate() == CampaignStatusPostmoderate.YES
                        && campaign.getStatusModerate() == CampaignStatusModerate.YES
                        && campaignService.needChangeStatusModeratePayCondition(container.getClientId(),
                        campaign.getId())
                ) {
                    changes.modify(CommonCampaign.STATUS_MODERATE, CampaignStatusModerate.READY);
                }
            }

            if (isValidId(changes.getOldValue(CommonCampaign.MANAGER_UID))
                    || isValidId(changes.getOldValue(CommonCampaign.AGENCY_UID))) {
                // allow pay without blocking for serviced campaigns
                changes.modify(CommonCampaign.STATUS_POST_MODERATE, CampaignStatusPostmoderate.ACCEPTED);
            }

            // проставляем дефолтные значения
            CampaignConverter.modifyFieldIfNull(changes, CommonCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT,
                    CampaignConstants.DEFAULT_ENABLE_PAUSED_BY_DAY_BUDGET_EVENT);

            // проставляем старое значение, чтобы не перезатирать
            CampaignConverter.setOldValueIfNewNull(changes, CommonCampaign.ENABLE_SEND_ACCOUNT_NEWS);
            CampaignConverter.setOldValueIfNewNull(changes, CommonCampaign.WARNING_BALANCE);
            CampaignConverter.setOldValueIfNewNull(changes, CommonCampaign.HAS_TURBO_APP);

            CampaignConverter.setOldValueIfNewNull(changes, CommonCampaign.IS_SERVICE_REQUESTED);
        });
    }

    @Override
    public void addToAdditionalActionsContainer(CampaignAdditionalActionsContainer additionalActionsContainer,
                                                RestrictedCampaignsUpdateOperationContainer updateParameters,
                                                List<AppliedChanges<CommonCampaign>> appliedChanges) {
        //noinspection ConstantConditions
        List<CampaignEvent<String>> campaignChangedStartDateEvents = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(CommonCampaign.START_DATE))
                .map(ac -> CampaignEvent.changedStartDateEvent(
                        updateParameters.getOperatorUid(),
                        updateParameters.getChiefUid(),
                        ac.getModel().getId(),
                        ac.getOldValue(CommonCampaign.START_DATE), ac.getNewValue(CommonCampaign.START_DATE)))
                .toList();
        additionalActionsContainer.addMailEventsToQueue(campaignChangedStartDateEvents);

        String textForEndDateNullValue = mailTextCreatorService.getTextForEndDateNullValue();
        List<CampaignEvent<String>> campaignChangedEndDateEvents = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(CommonCampaign.END_DATE))
                .map(ac -> CampaignEvent.changedEndDateEvent(
                        updateParameters.getOperatorUid(),
                        updateParameters.getChiefUid(),
                        ac.getModel().getId(), textForEndDateNullValue,
                        ac.getOldValue(CommonCampaign.END_DATE), ac.getNewValue(CommonCampaign.END_DATE)))
                .toList();
        additionalActionsContainer.addMailEventsToQueue(campaignChangedEndDateEvents);
    }

    @Override
    public void afterExecution(RestrictedCampaignsUpdateOperationContainer container,
                               List<AppliedChanges<CommonCampaign>> appliedChanges) {
        Optional<Long> managerUid = appliedChanges.stream()
                .filter(c -> c.assigned(CommonCampaign.MANAGER_UID))
                .findAny() // так как manager один на все кампании клиента
                .map(c -> c.getNewValue(CommonCampaign.MANAGER_UID));
        if (managerUid.isEmpty()) {
            return;
        }
        clientService.bindClientsToManager(managerUid.get(), List.of(container.getClientId()));

        if (container.getOptions().isSendAutoServicingMailNotification()) {
            var client = userService.getUser(container.getClientUid());
            var manager = userService.getUser(managerUid.get());
            appliedChanges.stream()
                    .filter(c -> c.assigned(CommonCampaign.MANAGER_UID))
                    .forEach(c -> sendAutoServicingMailNotification(
                            c.getModel(),
                            Objects.requireNonNull(client),
                            Objects.requireNonNull(manager)));
        }
    }

    private void sendAutoServicingMailNotification(
            CommonCampaign campaign,
            User subjectUser,
            User manager) {
        notificationService.addNotification(
                new AutoServicingMailNotification(campaign.getId(), campaign.getName(), campaign.getType(),
                        subjectUser, manager));
    }
}
