package ru.yandex.direct.core.entity.campaign.service.type.update;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils;
import ru.yandex.direct.model.AppliedChanges;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.hasChangesInStrategyToChangeDailyChangeCount;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isCpmStrategyShouldNotRestart;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isStrategyAlreadyStartedAndNotFinished;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isStrategyNameForCpmStrategyWithCustomPeriod;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@ParametersAreNonnullByDefault
public class CpmCampaignWithCustomStrategyUpdateOperationSupport extends AbstractCampaignUpdateOperationSupport<CpmCampaignWithCustomStrategy> {
    @Override
    public Class<CpmCampaignWithCustomStrategy> getTypeClass() {
        return CpmCampaignWithCustomStrategy.class;
    }

    @Override
    public void onChangesApplied(RestrictedCampaignsUpdateOperationContainer updateContainer,
                                 List<AppliedChanges<CpmCampaignWithCustomStrategy>> appliedChanges) {
        LocalDateTime now = now();
        StreamEx.of(appliedChanges)
                .mapToEntry(x -> x.getNewValue(CpmCampaignWithCustomStrategy.STRATEGY))
                .mapValues(DbStrategyBase::getStrategyData)
                .filterValues(CampaignWithStrategyValidationUtils::isStrategyNameForCpmStrategyWithCustomPeriod)
                .keys()
                .forEach(ac -> modifyStrategyData(now, ac));
        super.onChangesApplied(updateContainer, appliedChanges);
    }

    private void modifyStrategyData(LocalDateTime now, AppliedChanges<CpmCampaignWithCustomStrategy> ac) {
        StrategyData oldStrategyData =
                ac.getOldValue(CpmCampaignWithCustomStrategy.STRATEGY).getStrategyData();
        StrategyData newStrategyData =
                ac.getNewValue(CpmCampaignWithCustomStrategy.STRATEGY).getStrategyData();

        if (dailyChangeCounterShouldIncrease(now, oldStrategyData, newStrategyData)) {
            DbStrategy newStrategy = ac.getNewValue(CpmCampaignWithCustomStrategy.STRATEGY);
            newStrategy.getStrategyData()
                    .withLastUpdateTime(now)
                    .withDailyChangeCount(nvl(oldStrategyData.getDailyChangeCount(), 0L) + 1);
            ac.modify(CpmCampaignWithCustomStrategy.STRATEGY, newStrategy);
        } else if (dailyChangeCounterShouldReset(now, oldStrategyData, newStrategyData)) {
            DbStrategy newStrategy = ac.getNewValue(CpmCampaignWithCustomStrategy.STRATEGY);
            newStrategy.getStrategyData()
                    .withLastUpdateTime(now)
                    .withDailyChangeCount(1L);
            ac.modify(CpmCampaignWithCustomStrategy.STRATEGY, newStrategy);
        } else if (isStrategyNameForCpmStrategyWithCustomPeriod(newStrategyData)) {
            DbStrategy oldStrategy = ac.getOldValue(CpmCampaignWithCustomStrategy.STRATEGY);
            DbStrategy newStrategy = ac.getNewValue(CpmCampaignWithCustomStrategy.STRATEGY);

            newStrategy.getStrategyData()
                    .withLastUpdateTime(oldStrategy.getStrategyData().getLastUpdateTime())
                    .withDailyChangeCount(oldStrategy.getStrategyData().getDailyChangeCount());
            ac.modify(CpmCampaignWithCustomStrategy.STRATEGY, newStrategy);
        }
    }

    private static boolean dailyChangeCounterShouldReset(LocalDateTime now, StrategyData oldStrategyData,
                                                         StrategyData newStrategyData) {
        return isStrategyNameForCpmStrategyWithCustomPeriod(newStrategyData) &&
                (!isCpmStrategyShouldNotRestart(oldStrategyData, newStrategyData) ||
                        (!isCampaignStrategyLastChangeWasToday(now, oldStrategyData) &&
                                hasChangesInStrategyToChangeDailyChangeCount(oldStrategyData, newStrategyData)));
    }

    private static boolean dailyChangeCounterShouldIncrease(LocalDateTime now, StrategyData oldStrategyData,
                                                            StrategyData newStrategyData) {
        return isCpmStrategyShouldNotRestart(oldStrategyData, newStrategyData) &&
                isCampaignStrategyLastChangeWasToday(now, oldStrategyData) &&
                isStrategyAlreadyStartedAndNotFinished(oldStrategyData, now.toLocalDate()) &&
                hasChangesInStrategyToChangeDailyChangeCount(oldStrategyData, newStrategyData);
    }

    private static boolean isCampaignStrategyLastChangeWasToday(LocalDateTime now, StrategyData oldStrategyData) {
        return oldStrategyData.getLastUpdateTime() != null &&
                now.toLocalDate().equals(oldStrategyData.getLastUpdateTime().toLocalDate());
    }
}
