package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmDealsCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmYndxFrontpageCampaign;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.service.validation.type.add.CampaignAddValidationTypeSupportFacade;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.validation.defects.RightsDefects;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.UserPerminfo;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

@Service
@ParametersAreNonnullByDefault
public class AddRestrictedCampaignValidationService {
    private final static Set<Class<? extends BaseCampaign>> SOCIAL_CAMP_CLASSES = Set.of(
            CpmPriceCampaign.class, TextCampaign.class
    );
    private final static Set<Class<? extends  BaseCampaign>> CPM_CAMP_CLASSES = Set.of(
            CpmPriceCampaign.class, CpmBannerCampaign.class, CpmDealsCampaign.class, CpmYndxFrontpageCampaign.class
    );
    private final FeatureService featureService;
    private final RbacService rbacService;
    private final CampaignAddValidationTypeSupportFacade typeSupportFacade;
    private final CampaignLimitsValidationService campaignLimitsValidationService;

    @Autowired
    public AddRestrictedCampaignValidationService(FeatureService featureService, RbacService rbacService,
                                                  CampaignAddValidationTypeSupportFacade typeSupportFacade,
                                                  CampaignLimitsValidationService campaignLimitsValidationService) {
        this.featureService = featureService;
        this.rbacService = rbacService;
        this.typeSupportFacade = typeSupportFacade;
        this.campaignLimitsValidationService = campaignLimitsValidationService;
    }

    public ValidationResult<? extends List<? extends BaseCampaign>, Defect> preValidate(
            CampaignValidationContainer container,
            Long clientUid,
            List<? extends BaseCampaign> campaigns) {
        var vb = ListValidationBuilder.of(campaigns, Defect.class);

        Set<String> enabledFeatures = featureService.getEnabledForClientId(container.getClientId());
        vb.checkEach(Constraint.fromPredicate(campaign -> campaignTypeIsAvailableToAdd(campaign, enabledFeatures),
                CampaignDefects.campaignTypeNotSupported()));

        UserPerminfo clientPermInfo = rbacService.getUserPermInfo(clientUid);

        vb.check(Constraint.fromPredicate(campaignList ->
                        !rbacService.getUserPermInfo(container.getOperatorUid()).hasRole(RbacRole.MEDIA),
                RightsDefects.noRights()));

        vb.check(Constraint.fromPredicate(campaignList -> isRealClient(clientPermInfo), RightsDefects.noRights()),
                When.isValid());

        boolean canWrite = rbacService.canWrite(container.getOperatorUid(), clientUid);

        vb.check(Constraint.fromPredicate(campaignList -> canWrite, RightsDefects.noRights()), When.isValid());

        var vr = vb.getResult();

        campaignLimitsValidationService.campaignsCountLimitCheck(container.getShard(),
                container.getClientId(), campaigns, vr);

        typeSupportFacade.preValidate(container, vr);

        return vr;
    }

    private boolean campaignTypeIsAvailableToAdd(BaseCampaign campaign, Set<String> enabledFeatures) {
        return (!enabledFeatures.contains(FeatureName.SOCIAL_ADVERTISING.getName())
                || SOCIAL_CAMP_CLASSES.contains(campaign.getClass())) &&
                (!enabledFeatures.contains(FeatureName.IS_CPM_BANNER_CAMPAIGN_DISABLED.getName())
                        || !CPM_CAMP_CLASSES.contains(campaign.getClass()));
    }

    private boolean isRealClient(UserPerminfo clientPermInfo) {
        return clientPermInfo.role().anyOf(RbacRole.CLIENT, RbacRole.EMPTY);
    }

    public ValidationResult<? extends List<? extends BaseCampaign>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<? extends List<? extends BaseCampaign>, Defect> vr) {

        typeSupportFacade.validate(container, vr);
        return vr;
    }

}
