package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignAlreadyArchived;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignWithMoneyArchiving;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.nonStoppedCampaignArchiving;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class ArchiveCampaignValidationService {
    private static final int MAX_ELEMENTS_PER_OPERATION = 1000;
    private static final int MINUTES_AFTER_LAST_SHOW_FOR_ARC_CAMP = 60;

    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ModelChangesValidationTool preValidationTool;

    @Autowired
    public ArchiveCampaignValidationService(
            CampaignRepository campaignRepository,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory
    ) {
        this.campaignRepository = campaignRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.preValidationTool = ModelChangesValidationTool.builder()
                .objectNotFoundDefect(campaignNotFound())
                .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION).build();
    }

    public ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChanges(
            List<ModelChanges<Campaign>> modelChangesList,
            long operatorUid, ClientId clientId, int shard
    ) {
        List<Long> campaignIds = mapList(modelChangesList, ModelChanges::getId);
        Set<Long> existingCampaignIds = new HashSet<>(campaignRepository.getExistingCampaignIds(shard, clientId,
                campaignIds));

        CampaignSubObjectAccessValidator validator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, existingCampaignIds)
                .createValidator(CampaignAccessType.READ_WRITE);

        ValidationResult<List<ModelChanges<Campaign>>, Defect> validationResult =
                preValidationTool.validateModelChangesList(modelChangesList, existingCampaignIds);

        return new ListValidationBuilder<>(validationResult)
                .checkEachBy(changes -> {
                    ItemValidationBuilder<ModelChanges<Campaign>, Defect> ivb = ItemValidationBuilder.of(changes);
                    ivb.item(changes.getId(), Campaign.ID.name()).checkBy(validator, When.isValid());
                    return ivb.getResult();
                }, When.isValid())
                .getResult()
                .transformUnchecked(new Transformer());
    }

    public ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChangesBeforeApply(
            int shard,
            ValidationResult<List<ModelChanges<Campaign>>, Defect> preValidateResult,
            Map<Long, Campaign> campaigns
    ) {
        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(isNotEmpty(campaigns), When.isValid())
                .checkEach(isNoMoneyLeft(campaigns), When.isValid())
                .checkEach(isNotWalletType(campaigns), When.isValid())
                .checkEach(isNotBillingAggregateType(campaigns), When.isValid())
                .checkEach(isStopped(shard, campaigns), When.isValid())
                .weakCheckEach(isNotArchived(campaigns), When.isValid())
                .getResult();
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNotEmpty(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) && !campaigns.get(mc.getId()).getStatusEmpty(),
                campaignNotFound());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNotArchived(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) && !campaigns.get(mc.getId()).getStatusArchived(),
                campaignAlreadyArchived());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNoMoneyLeft(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) &&
                        !(campaigns.get(mc.getId()).getSum().subtract(campaigns.get(mc.getId()).getSumSpent())
                                .compareTo(Currencies.EPSILON) >= 0),
                campaignWithMoneyArchiving());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNotWalletType(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) &&
                        !(campaigns.get(mc.getId()).getType().equals(CampaignType.WALLET)),
                campaignTypeNotSupported());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNotBillingAggregateType(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) &&
                        !(campaigns.get(mc.getId()).getType().equals(CampaignType.BILLING_AGGREGATE)),
                campaignTypeNotSupported());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isStopped(int shard, Map<Long, Campaign> campaigns) {
        List<Long> campaignIds = mapList(campaigns.values(), Campaign::getId);

        Set<Long> stoppedCampaigns = campaignRepository.getStoppedCampaignsForArchiving(shard, campaignIds,
                MINUTES_AFTER_LAST_SHOW_FOR_ARC_CAMP);
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) &&
                        (stoppedCampaigns.contains(mc.getId()) ||
                                campaigns.get(mc.getId()).getStatusModerate().equals(CampaignStatusModerate.NEW) ||
                                !CampaignTypeKinds.WEB_EDIT_BASE.contains(campaigns.get(mc.getId()).getType())),
                nonStoppedCampaignArchiving());
    }

    @ParametersAreNonnullByDefault
    private static class Transformer implements ValidationResult.ValidationResultTransformer<Defect> {
        @Override
        public List<Defect> transformErrors(@SuppressWarnings("unused") Path path, List<Defect> errors) {
            return filterList(errors,
                    error -> !error.defectId().equals(CampaignDefectIds.Gen.ARCHIVED_CAMPAIGN_MODIFICATION));
        }
    }
}
