package ru.yandex.direct.core.entity.campaign.service.validation;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.net.InternetDomainName;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisabledDomainsAndSsp;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.net.IpUtils.ipFromString;
import static ru.yandex.direct.common.net.IpUtils.isPrivateAddress;
import static ru.yandex.direct.common.net.IpUtils.isValidIp;
import static ru.yandex.direct.core.entity.campaign.model.CampaignsDayBudgetShowMode.toSource;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_DAY_BUDGET_DAILY_CHANGE_COUNT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNoRights;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.dayBudgetOverridenByWallet;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.dayBudgetShowModeOverridenByWallet;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.duplicatedStrings;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.invalidDisabledDomain;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.invalidIpFormat;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.ipCantBeFromInternalNetwork;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.ipCantBeFromPrivateNetwork;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.tooManyDayBudgetDailyChanges;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.unknownSsp;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class CampaignConstraints {
    private CampaignConstraints() {
    }

    public static <C> Constraint<C, Defect> metrikaReturnsResultWithErrors() {
        return Constraint.fromPredicate(campaignModelChanges -> false, Defects.metrikaReturnsResultWithErrors());
    }


    public static Constraint<Long, Defect> campaignPresents(Set<Long> existingCampaigns) {
        return Constraint.fromPredicate(Predicates.inSet(existingCampaigns), campaignNotFound());
    }

    public static Constraint<Long, Defect> campaignIsNotArchived(Set<Long> archivedCampaigns) {
        return Constraint.fromPredicate(Predicates.notInSet(archivedCampaigns), archivedCampaignModification());
    }

    public static Constraint<Long, Defect> campaignIsVisible(Set<Long> visibleCampaigns) {
        return Constraint.fromPredicate(Predicates.inSet(visibleCampaigns), campaignNotFound());
    }

    public static Constraint<Long, Defect> campaignIsWritable(Set<Long> writableCampaigns) {
        return Constraint.fromPredicate(Predicates.inSet(writableCampaigns), campaignNoRights());
    }

    public static Constraint<String, Defect> disabledIpIsValid() {
        return ip -> isValidIp(ip) ? null : invalidIpFormat(ip);
    }

    public static Constraint<String, Defect> disabledIpIsNotPrivate() {
        return ip -> !isPrivateAddress(ipFromString(ip)) ? null : ipCantBeFromPrivateNetwork(ip);
    }

    public static Constraint<String, Defect> disabledIpIsNotInternal(NetAcl networks) {
        return ip -> !networks.isInternalIp(ipFromString(ip)) ? null : ipCantBeFromInternalNetwork(ip);
    }

    public static Constraint<CampaignWithDisabledDomainsAndSsp, Defect> totalDisabledPlacesMaxSize(int sizeLimit) {
        return Constraint.fromPredicate(
                c -> checkDisabledPlacesMaxSize(c, sizeLimit), CollectionDefects.maxCollectionSize(sizeLimit));
    }

    public static boolean checkDisabledPlacesMaxSize(CampaignWithDisabledDomainsAndSsp campaign, int sizeLimit) {
        return Stream.concat(
                nvl(campaign.getDisabledDomains(), emptyList()).stream(),
                nvl(campaign.getDisabledSsp(), emptyList()).stream()
        )
                .collect(Collectors.toSet())
                .size() <= sizeLimit;
    }

    public static Constraint<String, Defect> isValidDomain() {
        return domain -> {
            try {
                if (InternetDomainName.from(domain).parts().size() > 1) {
                    return null;
                } else {
                    return invalidDisabledDomain(domain);
                }
            } catch (IllegalArgumentException e) {
                return invalidDisabledDomain(domain);
            }
        };
    }

    public static Constraint<String, Defect> isValidDisabledDomain(ClientId clientId, DisableDomainValidationService disableDomainValidationService) {
        return d -> disableDomainValidationService.isValidDomain(d, clientId) ? null :
                invalidDisabledDomain(d);
    }

    public static Constraint<String, Defect> isValidSsp(Set<String> knownSsp) {
        return ssp -> knownSsp.contains(ssp) ? null : unknownSsp(ssp);
    }

    public static Constraint<BigDecimal, Defect> dayBudgetCanBeSet(Integer dayBudgetDailyChangeCount) {
        return Constraint.fromPredicate(dayBudget -> NumberUtils.isZero(dayBudget)
                        //dayBudgetDailyChangeCount может быть null при создании кампании
                        || null == dayBudgetDailyChangeCount
                        || dayBudgetDailyChangeCount <= MAX_DAY_BUDGET_DAILY_CHANGE_COUNT,
                tooManyDayBudgetDailyChanges());
    }

    public static Constraint<BigDecimal, Defect> isDayBudgetOverridenByWallet(Campaign wallet) {
        return Constraint.fromPredicate(dayBudget -> NumberUtils.isZero(dayBudget)
                || wallet.getStrategy().getDayBudget().compareTo(dayBudget) >= 0, dayBudgetOverridenByWallet());
    }

    public static Constraint<DayBudgetShowMode, Defect> isDayBudgetShowModeOverridenByWallet(Campaign wallet) {
        return Constraint.fromPredicate(dayBudgetShowMode -> toSource(wallet.getStrategy().getDayBudgetShowMode())
                        == DayBudgetShowMode.toSource(dayBudgetShowMode),
                dayBudgetShowModeOverridenByWallet());
    }

    public static Constraint<CampaignWithDisabledDomainsAndSsp, Defect> uniquePlaces() {
        return campaign -> {
            List<String> duplicated = new ArrayList<>();
            List<String> duplicatedSsp = getDuplicatedStrings(campaign.getDisabledSsp());
            List<String> duplicatedDomains = getDuplicatedStrings(campaign.getDisabledDomains());
            ifNotNull(duplicatedSsp, duplicated::addAll);
            ifNotNull(duplicatedDomains, duplicated::addAll);
            if (!duplicated.isEmpty()) {
                return duplicatedStrings(duplicated);
            }
            return null;
        };
    }

    public static Constraint<List<String>, Defect> uniqueStrings() {
        return strings -> {
            List<String> duplicated = getDuplicatedStrings(strings);
            return CollectionUtils.isEmpty(duplicated) ? null : duplicatedStrings(duplicated);
        };
    }

    @Nullable
    private static List<String> getDuplicatedStrings(@Nullable List<String> strings) {
        if (strings == null) {
            return null;
        }
        return strings.stream()
                .collect(groupingBy(identity(), counting()))
                .entrySet().stream()
                .filter(e -> e.getValue() > 1L)
                .map(Map.Entry::getKey)
                .collect(toList());
    }

}
