package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.IntStreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignCounts;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.unarcCampaignsLimitExceeded;
import static ru.yandex.direct.validation.result.PathHelper.index;

@Service
@ParametersAreNonnullByDefault
public class CampaignLimitsValidationService {
    private final ClientLimitsService clientLimitsService;
    private final CampaignRepository campaignRepository;

    public CampaignLimitsValidationService(ClientLimitsService clientLimitsService,
                                           CampaignRepository campaignRepository) {
        this.clientLimitsService = clientLimitsService;
        this.campaignRepository = campaignRepository;
    }

    public void campaignsCountLimitCheck(
            int shard, ClientId clientId, List<? extends BaseCampaign> campaigns,
            ValidationResult<? extends List<?>, Defect> vr) {

        ClientLimits clientLimits = clientLimitsService.getClientLimits(clientId);
        CampaignCounts campaignCounts = campaignRepository.getCampaignCountsForCountLimitCheck(
                shard, clientId.asLong());

        Long maxCountOfCampaigns = clientLimits.getCampsCountLimitOrDefault();
        Long maxCountOfUnarchivedCampaigns = clientLimits.getUnarcCampsCountLimitOrDefault();

        if (campaignCounts.getUnarchived() + campaigns.size() > maxCountOfUnarchivedCampaigns) {
            vr.addError(CampaignDefects.maxCampaignsForClientId(maxCountOfUnarchivedCampaigns));
        } else if (campaignCounts.getAll() + campaigns.size() > maxCountOfCampaigns) {
            vr.addError(CampaignDefects.maxCampaignsForClientId(maxCountOfCampaigns));
        }
    }

    public void unarcCampaignsCountLimitCheck(
            int shard, ClientId clientId,
            ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> vr
    ) {
        ClientLimits clientLimits = clientLimitsService.getClientLimits(clientId);
        CampaignCounts campaignCounts = campaignRepository.getCampaignCountsForCountLimitCheck(
                        shard, clientId.asLong());

        Long maxCountOfUnarchivedCampaigns = clientLimits.getUnarcCampsCountLimitOrDefault();
        Long campaignsLeftToUnarchive = maxCountOfUnarchivedCampaigns - campaignCounts.getUnarchived();
        Set<Long> campIdsOutsideLimit = IntStreamEx.range(0, vr.getValue().size())
                // рассматриваем в том порядке, в каком задал пользователь
                .mapToObj(i -> vr.getOrCreateSubValidationResult(index(i), vr.getValue().get(i)))
                .filter(e -> !e.hasAnyErrors())
                .skip(campaignsLeftToUnarchive)
                .map(e -> e.getValue().getId())
                .toSet();
        new ListValidationBuilder<>(vr)
                .checkEach(Constraint.fromPredicate(
                                mc -> !campIdsOutsideLimit.contains(mc.getId()),
                                unarcCampaignsLimitExceeded(maxCountOfUnarchivedCampaigns)),
                        When.isValid());
    }
}
