package ru.yandex.direct.core.entity.campaign.service.validation;

import java.time.LocalDate;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.defect.DateDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;

import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@ParametersAreNonnullByDefault
public class CampaignValidators {
    private CampaignValidators() {
    }

    public static <T extends ModelWithId> Validator<ModelChanges<T>, Defect> fieldCantBeModified(
            ModelProperty<? super T, ?> modelProperty) {
        return fieldCantBeModified(modelProperty, noRights());
    }

    public static <T extends ModelWithId> Validator<ModelChanges<T>, Defect> fieldCantBeModified(
            ModelProperty<? super T, ?> modelProperty, Defect<Void> defect) {
        return mc -> {
            if (!mc.isPropChanged(modelProperty)) {
                return ValidationResult.success(mc);
            }

            return new ValidationResult<>(mc, null, null,
                    Map.of(PathHelper.field(modelProperty),
                            ValidationResult.failed(mc.getChangedProp(modelProperty), defect)));
        };
    }

    public static Validator<ModelChanges<CommonCampaign>, Defect> dateNotInPastIfChanged(
            Map<Long, CommonCampaign> unmodifiedModels,
            ModelProperty<? super CommonCampaign, LocalDate> dateModelProperty) {
        return mc -> {
            CommonCampaign oldCampaign = unmodifiedModels.get(mc.getId());
            LocalDate now = LocalDate.now();
            var vb = ModelChangesValidationBuilder.of(mc);
            vb.item(dateModelProperty)
                    .check(fromPredicate(date ->
                                    date.equals(dateModelProperty.get(oldCampaign))
                                            || !date.isBefore(now),
                            DateDefects.greaterThanOrEqualTo(now)));
            return vb.getResult();
        };
    }
}
