package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithMetrikaCountersValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.UserCounters;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.SPRAV;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isUnavailableGoalsAllowed;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_NUMBER_OF_OPTIONAL_METRIKA_COUNTERS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.feature.FeatureName.ADDING_ORGANIZATIONS_COUNTERS_TO_CAMPAIGN_ON_ADDING_ORGANIZATIONS_TO_ADS;
import static ru.yandex.direct.feature.FeatureName.METRIKA_COUNTERS_ACCESS_VALIDATION_ON_SAVE_CAMPAIGN_ENABLED;

@Component
public class CampaignWithMetrikaCountersValidatorProvider {
    private static final Logger logger = LoggerFactory.getLogger(CampaignWithMetrikaCountersValidatorProvider.class);

    private final FeatureService featureService;

    private static final Validator<CampaignWithMetrikaCounters, Defect> METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR
            = campaign -> {
        ModelItemValidationBuilder<CampaignWithMetrikaCounters> ivb = ModelItemValidationBuilder.of(campaign);
        ivb.check(metrikaReturnsResultWithErrors(), When.notNull());
        return ivb.getResult();
    };

    @Autowired
    public CampaignWithMetrikaCountersValidatorProvider(FeatureService featureService) {
        this.featureService = featureService;
    }

    public Validator<CampaignWithMetrikaCounters, Defect> campaignWithMetrikaCountersValidator(
            CampaignValidationContainer container,
            List<CampaignWithMetrikaCounters> unvalidatedCampaigns) {

        Set<String> enabledFeatures = featureService.getEnabledForClientId(container.getClientId());
        boolean metrikaCountersAccessValidationEnabledByCampaign = unvalidatedCampaigns.stream()
                .anyMatch(campaign -> needToValidateMetrikaCountersAccess(campaign, enabledFeatures));
        boolean metrikaCountersAccessValidationEnabledByFeature = enabledFeatures.contains(
                METRIKA_COUNTERS_ACCESS_VALIDATION_ON_SAVE_CAMPAIGN_ENABLED.getName());

        Set<Long> availableCounterIds;
        try {
            availableCounterIds = getAvailableCounterIds(
                    container,
                    metrikaCountersAccessValidationEnabledByCampaign,
                    metrikaCountersAccessValidationEnabledByFeature
            );
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying available counters from metrika by clientId {}.",
                    container.getClientId(), e);
            return METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR;
        }

        boolean addingOrganizationsCountersToCampaignOnAddingOrganizationsToAds = enabledFeatures.contains(
                ADDING_ORGANIZATIONS_COUNTERS_TO_CAMPAIGN_ON_ADDING_ORGANIZATIONS_TO_ADS.getName());

        Set<Long> clientSystemCounterIds;
        try {
            clientSystemCounterIds = getClientSystemCounterIds(
                    container, addingOrganizationsCountersToCampaignOnAddingOrganizationsToAds
            );
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for organization counters from metrika by clientId {}. " +
                    "Exception: {}", container.getClientId(), e);
            return METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR;
        }

        return campaign -> CampaignWithMetrikaCountersValidator
                .build(availableCounterIds,
                        clientSystemCounterIds,
                        needToValidateMetrikaCountersAccess(campaign, enabledFeatures)
                                || metrikaCountersAccessValidationEnabledByFeature,
                        MAX_NUMBER_OF_OPTIONAL_METRIKA_COUNTERS,
                        campaign.getMinSizeOfMetrikaCounters())
                .apply(campaign);
    }

    private boolean needToValidateMetrikaCountersAccess(CampaignWithMetrikaCounters campaign,
                                                        Set<String> enabledFeatures) {
        return campaign.isMetrikaCountersAccessValidationRequired() &&
                !isUnavailableGoalsAllowed(campaign, enabledFeatures);
    }

    private Set<Long> getAvailableCounterIds(CampaignValidationContainer container,
                                             boolean isMetrikaCountersAccessValidationRequired,
                                             boolean metrikaCountersAccessValidationEnabledByFeature) {
        if (!isMetrikaCountersAccessValidationRequired && !metrikaCountersAccessValidationEnabledByFeature) {
            return Collections.emptySet();
        }

        return StreamEx.of(container.getMetrikaClient().getUsersCountersNumByCampaignCounterIds())
                .flatCollection(UserCounters::getCounterIds)
                .map(Integer::longValue)
                .toSet();
    }

    private Set<Long> getClientSystemCounterIds(CampaignValidationContainer container,
                                                boolean addingOrganizationsCountersToCampaignOnAddingOrganizationsToAds) {
        if (!addingOrganizationsCountersToCampaignOnAddingOrganizationsToAds) {
            return Collections.emptySet();
        }
        return StreamEx.of(container.getMetrikaClient().getUsersCountersNumExtendedByCampaignCounterIds())
                .flatCollection(UserCountersExtended::getCounters)
                .filterBy(CounterInfoDirect::getCounterSource, SPRAV)
                .map(CounterInfoDirect::getId)
                .map(Long::valueOf)
                .toSet();
    }
}
