package ru.yandex.direct.core.entity.campaign.service.validation;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacRole;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove.NEW;
import static ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove.NO;
import static ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove.YES;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.pricePackageAllowsImpressionRate;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.pricePackageHasImpressionRate;
import static ru.yandex.direct.utils.DateTimeUtils.inFuture;
import static ru.yandex.direct.utils.DateTimeUtils.inFutureOrToday;
import static ru.yandex.direct.utils.DateTimeUtils.inPast;

@ParametersAreNonnullByDefault
public class CampaignWithPricePackagePermissionUtils {

    private static final Map<PriceFlightStatusApprove, List<PriceFlightStatusApprove>>
            FINISHED_CAMPAIGN_ALLOWED_STATUS_CHANGE =
            Map.of(NEW, List.of(NO),
                    NO, List.of(),
                    YES, List.of());

    private static final Map<PriceFlightStatusApprove, List<PriceFlightStatusApprove>>
            NOT_STARTED_OR_IN_PROGRESS_CAMPAIGN_ALLOWED_STATUS_CHANGE =
            Map.of(NEW, List.of(NO, YES),
                    NO, List.of(YES),
                    YES, List.of(NO));

    public static boolean canApprovePriceSalesCampaigns(User operator) {
        return canApprovePriceSalesCampaigns(operator.getRole(), emptySet());
    }

    private static boolean canApprovePriceSalesCampaigns(RbacRole operatorRole, Set<String> features) {
        return operatorRole == RbacRole.SUPER || features.contains(FeatureName.IS_ABLE_TO_APPROVE_CAMPAIGNS.getName());
    }

    public static boolean canResetFlightStatusApprove(User operator) {
        return operator.getRole().anyOf(RbacRole.SUPPORT, RbacRole.SUPER, RbacRole.MANAGER);
    }

    public static boolean canChangeStartDate(CampaignWithPricePackage model,
                                             CampaignWithPricePackageValidateBeforeApplyContext context,
                                             LocalDate newDate) {

        User operator = context.getOperator();
        return model.getFlightStatusApprove() != YES
                || (canApprovePriceSalesCampaigns(context.getOperatorRole(), context.getOperatorFeatures()) && inFuture(model.getStartDate()))
                || (operator.getRole() == RbacRole.MANAGER && canResetFlightStatusApprove(operator));
    }

    public static boolean canChangeEndDate(CampaignWithPricePackage model,
                                           CampaignWithPricePackageValidateBeforeApplyContext context,
                                           LocalDate newDate) {
        User operator = context.getOperator();
        return model.getFlightStatusApprove() != YES
                || !isBookingPackage(model, context)//для заапрувленного видео можем поменять дату окончания и объем
                || (canApprovePriceSalesCampaigns(context.getOperatorRole(), context.getOperatorFeatures()))
                || (operator.getRole() == RbacRole.MANAGER && canResetFlightStatusApprove(operator));
    }

    public static boolean canChangeFlightOrderVolume(CampaignWithPricePackage model,
                                                     CampaignWithPricePackageValidateBeforeApplyContext context,
                                                     Long newFlightOrderVolume) {
        User operator = context.getOperator();
        return model.getFlightStatusApprove() != YES
                || !isBookingPackage(model, context)//для заапрувленного видео можем поменять дату окончания и объем
                || (canApprovePriceSalesCampaigns(context.getOperatorRole(), context.getOperatorFeatures()) && inFutureOrToday(model.getEndDate()))
                || (operator.getRole() == RbacRole.MANAGER && canResetFlightStatusApprove(operator));
    }

    public static boolean canChangeFlightStatusApprove(CampaignWithPricePackage model,
                                                       CampaignWithPricePackageValidateBeforeApplyContext context,
                                                       PriceFlightStatusApprove newFlightStatusApprove) {
        if (newFlightStatusApprove == NEW) {
            return canResetFlightStatusApprove(context.getOperator());
        } else {
            return canChangeFlightStatusApproveToYesOrNo(model, context, newFlightStatusApprove);
        }
    }

    private static boolean canChangeFlightStatusApproveToYesOrNo(
            CampaignWithPricePackage model,
            CampaignWithPricePackageValidateBeforeApplyContext context,
            PriceFlightStatusApprove newFlightStatusApprove) {
        Map<PriceFlightStatusApprove, List<PriceFlightStatusApprove>> statusChangeMap;
        if (inPast(model.getEndDate())) {
            statusChangeMap = FINISHED_CAMPAIGN_ALLOWED_STATUS_CHANGE;
        } else {
            statusChangeMap = NOT_STARTED_OR_IN_PROGRESS_CAMPAIGN_ALLOWED_STATUS_CHANGE;
        }
        return canApprovePriceSalesCampaigns(context.getOperatorRole(), context.getOperatorFeatures())
                && statusChangeMap.get(model.getFlightStatusApprove()).contains(newFlightStatusApprove)
                && (newFlightStatusApprove != PriceFlightStatusApprove.YES
                || campaignIsWaitingApprove(model, context.getCampaignsWaitingApprove()));
    }

    private static boolean campaignIsWaitingApprove(CampaignWithPricePackage campaignToTest,
                                                    List<CampaignWithPricePackage> campaignsWaitingApprove) {
        return campaignsWaitingApprove.stream()
                .anyMatch(campaignWaitingApprove -> campaignWaitingApprove.getId().equals(campaignToTest.getId()));
    }

    @SuppressWarnings({"PointlessBooleanExpression", "SimplifyBooleanExpression"})
    public static boolean canChangeStatusShow(CampaignWithPricePackage model,
                                              CampaignWithPricePackageValidateBeforeApplyContext context,
                                              Boolean newStatusShow) {
        if (newStatusShow == true) {
            return canResumeCampaign(context.getOperatorRole(), model.getEndDate());
        } else {
            return canStopCampaign(context.getOperatorRole(), model.getEndDate());
        }
    }

    public static boolean canResumeCampaign(RbacRole operatorRole, LocalDate endDate) {
        return operatorRole == RbacRole.SUPER
                || (operatorRole == RbacRole.MANAGER && !inPast(endDate));
    }

    public static boolean canStopCampaign(RbacRole operatorRole, LocalDate endDate) {
        return operatorRole.anyOf(RbacRole.SUPER, RbacRole.MANAGER)
                // открученную кампанию разрешаем останавливать всем, чтобы её можно было заархивировать
                || inPast(endDate);
    }

    /**
     * Параметры ограничения частоты показов нельзя менять, если в пакете выставлены фиксированные параметры,
     * или запрещается указывать это ограничение.
     */
    public static boolean canChangeImpressionRate(CampaignWithPricePackage model,
                                                  CampaignWithPricePackageValidateBeforeApplyContext context,
                                                  Integer newValue) {
        //В случае если ограничения заданы на пакете, то считаем валидными если в модели будет старое значение
        // и если придёт изменение на null. В этом случае значение не меняется.
        // На уровне кампании фронт может не знать верного значения ограничения частоты
        PricePackage pricePackage = context.getPricePackagesMap().get(model.getPricePackageId());
        return pricePackageAllowsImpressionRate(pricePackage)
                && !pricePackageHasImpressionRate(pricePackage);
    }

    /**
     * Проверяет что это кампания с подтверждением бронирования. Сейчас это кампании на главной
     */
    public static boolean isBookingPackage(CampaignWithPricePackage model,
                                           CampaignWithPricePackageValidateBeforeApplyContext context) {
        PricePackage pricePackage = context.getPricePackagesMap().get(model.getPricePackageId());
        return pricePackage == null || pricePackage.getAvailableAdGroupTypes() == null
                || pricePackage.getAvailableAdGroupTypes().isEmpty()
                || pricePackage.isFrontpagePackage();
    }
}
