package ru.yandex.direct.core.entity.campaign.service.validation;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName;

import static ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform.BOTH;
import static ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform.CONTEXT;
import static ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform.SEARCH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CPI_GOAL_ID;
import static ru.yandex.direct.feature.FeatureName.INCREASED_CPA_LIMIT_FOR_PAY_FOR_CONVERSION;
import static ru.yandex.direct.utils.NumberUtils.equalsByCompareTo;

@ParametersAreNonnullByDefault
public class CampaignWithStrategyValidationUtils {

    public static final Set<CampOptionsStrategy> ALL_CAMP_OPTIONS_STRATEGIES = EnumSet.allOf(CampOptionsStrategy.class);
    public static final Set<CampaignsPlatform> ALL_CAMPAIGNS_PLATFORMS = EnumSet.allOf(CampaignsPlatform.class);

    public static final Map<CampaignType, Set<StrategyName>> CAMPAIGN_TO_STRATEGY_TYPE =
            ImmutableMap.<CampaignType, Set<StrategyName>>builder()
                    .put(CampaignType.TEXT, Set.of(StrategyName.AUTOBUDGET_WEEK_BUNDLE,
                            StrategyName.AUTOBUDGET_AVG_CPA, StrategyName.AUTOBUDGET, StrategyName.AUTOBUDGET_ROI,
                            StrategyName.DEFAULT_, StrategyName.AUTOBUDGET_AVG_CLICK, StrategyName.AUTOBUDGET_CRR))
                    .put(CampaignType.DYNAMIC, Set.of(StrategyName.AUTOBUDGET_WEEK_BUNDLE,
                            StrategyName.AUTOBUDGET_AVG_CPA, StrategyName.AUTOBUDGET, StrategyName.AUTOBUDGET_ROI,
                            StrategyName.DEFAULT_, StrategyName.AUTOBUDGET_AVG_CLICK, StrategyName.AUTOBUDGET_CRR))
                    .put(CampaignType.PERFORMANCE, Set.of(StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER,
                            StrategyName.AUTOBUDGET_ROI, StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER,
                            StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP, StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
                            StrategyName.AUTOBUDGET, StrategyName.AUTOBUDGET_CRR))
                    .put(CampaignType.CPM_BANNER, Set.of(StrategyName.AUTOBUDGET_MAX_IMPRESSIONS,
                            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD, StrategyName.CPM_DEFAULT,
                            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD, StrategyName.AUTOBUDGET_MAX_REACH,
                            StrategyName.AUTOBUDGET_AVG_CPV, StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
                            StrategyName.AUTOBUDGET_AVG_CPA, StrategyName.AUTOBUDGET))
                    .put(CampaignType.CPM_DEALS, Set.of(StrategyName.AUTOBUDGET_MAX_IMPRESSIONS,
                            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD, StrategyName.CPM_DEFAULT,
                            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD, StrategyName.AUTOBUDGET_MAX_REACH,
                            StrategyName.AUTOBUDGET_AVG_CPV, StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD))
                    .put(CampaignType.CPM_YNDX_FRONTPAGE, Set.of(StrategyName.AUTOBUDGET_MAX_IMPRESSIONS,
                            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD, StrategyName.CPM_DEFAULT,
                            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD, StrategyName.AUTOBUDGET_MAX_REACH))
                    .put(CampaignType.MCB, Set.of(StrategyName.DEFAULT_, StrategyName.AUTOBUDGET_MEDIA))
                    .put(CampaignType.MCBANNER, Set.of(StrategyName.DEFAULT_, StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP))
                    .put(CampaignType.CONTENT_PROMOTION, Set.of(StrategyName.DEFAULT_,
                            StrategyName.AUTOBUDGET_WEEK_BUNDLE, StrategyName.AUTOBUDGET,
                            StrategyName.AUTOBUDGET_AVG_CLICK))
                    .put(CampaignType.MOBILE_CONTENT, Set.of(StrategyName.AUTOBUDGET_WEEK_BUNDLE,
                            StrategyName.AUTOBUDGET, StrategyName.AUTOBUDGET_AVG_CPI, StrategyName.DEFAULT_,
                            StrategyName.AUTOBUDGET_AVG_CLICK, StrategyName.AUTOBUDGET_ROI,
                            StrategyName.AUTOBUDGET_AVG_CPA, StrategyName.AUTOBUDGET_CRR))
                    .put(CampaignType.INTERNAL_AUTOBUDGET, Set.of(StrategyName.AUTOBUDGET,
                            StrategyName.AUTOBUDGET_AVG_CLICK,
                            StrategyName.AUTOBUDGET_AVG_CPA,
                            StrategyName.AUTOBUDGET_WEEK_BUNDLE))
                    .put(CampaignType.INTERNAL_DISTRIB, Set.of(StrategyName.DEFAULT_))
                    .put(CampaignType.INTERNAL_FREE, Set.of(StrategyName.DEFAULT_))
                    // TODO: для прайсовых должно быть что-то другое, пока что так, чтобы работали
                    //       хоть какие-то тесты
                    .put(CampaignType.CPM_PRICE, Set.of(StrategyName.DEFAULT_))
                    .build();

    public static final Set<CampaignType> CAMPAIGN_TYPES_AVAILABLE_FOR_PAY_FOR_CONVERSION_EXTENDED_MODE =
            ImmutableSet.of(CampaignType.TEXT, CampaignType.DYNAMIC, CampaignType.PERFORMANCE,
                    CampaignType.MOBILE_CONTENT);

    // TODO: заполнить для остальных типов кампаний
    public static final Map<CampaignType, Set<CampaignsPlatform>> CAMPAIGN_TO_PLATFORM_TYPE =
            ImmutableMap.<CampaignType, Set<CampaignsPlatform>>builder()
                    .put(CampaignType.CONTENT_PROMOTION, Set.of(SEARCH))
                    .put(CampaignType.CPM_YNDX_FRONTPAGE, Set.of(CONTEXT))
                    .put(CampaignType.CPM_BANNER, Set.of(CONTEXT))
                    .put(CampaignType.PERFORMANCE, Set.of(SEARCH, CONTEXT, BOTH))
                    .put(CampaignType.DYNAMIC, Set.of(SEARCH, CONTEXT, BOTH))
                    .put(CampaignType.MOBILE_CONTENT, Set.of(SEARCH, CONTEXT, BOTH))
                    .put(CampaignType.MCBANNER, Set.of(SEARCH))
                    .build();

    // TODO: заполнить для остальных типов кампаний
    public static final Map<CampaignType, Set<CampOptionsStrategy>> CAMPAIGN_TO_CONTEXT_STRATEGY_TYPE =
            ImmutableMap.<CampaignType, Set<CampOptionsStrategy>>builder()
                    .put(CampaignType.CONTENT_PROMOTION, Set.of())
                    .put(CampaignType.CPM_YNDX_FRONTPAGE, Set.of(CampOptionsStrategy.DIFFERENT_PLACES))
                    .put(CampaignType.PERFORMANCE, Set.of(
                            CampOptionsStrategy.DIFFERENT_PLACES,
                            CampOptionsStrategy.AUTOBUDGET_AVG_CPC_PER_CAMP,
                            CampOptionsStrategy.AUTOBUDGET_AVG_CPC_PER_FILTER,
                            CampOptionsStrategy.AUTOBUDGET_AVG_CPA_PER_CAMP,
                            CampOptionsStrategy.AUTOBUDGET_AVG_CPA_PER_FILTER))
                    .put(CampaignType.MOBILE_CONTENT, Set.of(CampOptionsStrategy.DIFFERENT_PLACES))
                    .put(CampaignType.MCBANNER, Set.of(CampOptionsStrategy.AUTOBUDGET_AVG_CPC_PER_CAMP))
                    .build();

    public static final Set<String> STRATEGY_TYPES_WITH_LEARNING_STATUS = Set.of(
            CampaignsStrategyName.autobudget.getLiteral(),
            CampaignsStrategyName.autobudget_avg_cpa.getLiteral(),
            CampaignsStrategyName.autobudget_avg_cpa_per_camp.getLiteral(),
            CampaignsStrategyName.autobudget_avg_cpa_per_filter.getLiteral(),
            CampaignsStrategyName.autobudget_avg_cpi.getLiteral(),
            CampaignsStrategyName.autobudget_crr.getLiteral()
    );

    public static final Set<String> STRATEGY_WITH_REVENUE_MODEL = Set.of(
            CampaignsStrategyName.autobudget_crr.getLiteral()
    );

    /*
     *  Метод проверяет поменялась ли модель стратегии с доходной на недоходную.
     * */
    public static boolean isStrategyModelChanged(StrategyData oldStrategy, StrategyData newStrategy) {
        return STRATEGY_WITH_REVENUE_MODEL.contains(oldStrategy.getName()) ^
                STRATEGY_WITH_REVENUE_MODEL.contains(newStrategy.getName());
    }

    public static boolean isCpmStrategyWithCustomPeriodChangedForCheckStartDate(
            StrategyData strategyDataOld,
            StrategyData strategyDataNew) {
        boolean excludingFieldsChanged = !Objects.equals(strategyDataOld.getAutoProlongation(),
                strategyDataNew.getAutoProlongation()) ||
                !equalsByCompareTo(strategyDataOld.getAvgCpm(), strategyDataNew.getAvgCpm()) ||
                !equalsByCompareTo(strategyDataOld.getBudget(), strategyDataNew.getBudget());
        return (!isStrategyNameForCpmStrategyWithCustomPeriod(strategyDataNew) && excludingFieldsChanged) ||
                !Objects.equals(strategyDataOld.getName(), strategyDataNew.getName()) ||
                !Objects.equals(strategyDataOld.getStart(), strategyDataNew.getStart());
    }

    public static boolean isCpmStrategyWithCustomPeriodChangedForCheckFinishDate(
            StrategyData strategyDataOld,
            StrategyData strategyDataNew) {
        boolean excludingFieldsChanged = !Objects.equals(strategyDataOld.getAutoProlongation(),
                strategyDataNew.getAutoProlongation());
        return (!isStrategyNameForCpmStrategyWithCustomPeriod(strategyDataNew) &&
                excludingFieldsChanged) ||
                !Objects.equals(strategyDataOld.getName(), strategyDataNew.getName()) ||
                !equalsByCompareTo(strategyDataOld.getAvgCpm(), strategyDataNew.getAvgCpm()) ||
                !equalsByCompareTo(strategyDataOld.getBudget(), strategyDataNew.getBudget()) ||
                !Objects.equals(strategyDataOld.getStart(), strategyDataNew.getStart()) ||
                !Objects.equals(strategyDataOld.getFinish(), strategyDataNew.getFinish());
    }

    public static boolean isCpmStrategyShouldNotRestart(
            StrategyData strategyDataOld,
            StrategyData strategyDataNew) {
        return Objects.equals(strategyDataOld.getName(), strategyDataNew.getName()) &&
                isStrategyNameForCpmStrategyWithCustomPeriod(strategyDataNew) &&
                Objects.equals(strategyDataOld.getStart(), strategyDataNew.getStart());
    }

    public static boolean isCpiStrategyWithDefaultGoalId(DbStrategyBase strategy) {
        return strategy.getStrategyName() == StrategyName.AUTOBUDGET_AVG_CPI
                && DEFAULT_CPI_GOAL_ID.equals(strategy.getStrategyData().getGoalId());
    }

    public static boolean isCpiStrategy(StrategyData strategyData) {
        return CampaignsStrategyName.autobudget_avg_cpi.getLiteral().equals(strategyData.getName());
    }

    public static boolean hasChangesInStrategyToChangeDailyChangeCount(StrategyData strategyDataOld,
                                                                       StrategyData strategyDataNew) {
        return !(equalsByCompareTo(strategyDataOld.getBudget(), strategyDataNew.getBudget()) &&
                equalsByCompareTo(strategyDataOld.getAvgCpm(), strategyDataNew.getAvgCpm()) &&
                Objects.equals(strategyDataOld.getFinish(), strategyDataNew.getFinish()));
    }

    public static boolean isStrategyNameForCpmStrategyWithCustomPeriod(StrategyData strategyData) {
        if (strategyData.getName() == null) {
            return false;
        }
        return Set.of(CampaignsStrategyName.autobudget_avg_cpv_custom_period.getLiteral(),
                CampaignsStrategyName.autobudget_max_impressions_custom_period.getLiteral(),
                CampaignsStrategyName.autobudget_max_reach_custom_period.getLiteral()).contains(strategyData.getName());
    }

    public static boolean isStrategyWithSupportOfLearningStatus(StrategyData strategyData) {
        if (strategyData.getName() == null) {
            return false;
        }
        if (isCpiStrategy(strategyData)) {
            // не проверяем цель, т.к. цель или присутствует или может быть null
            // если используется дефолтная цель для CPI стратегии
            return true;
        }
        return strategyData.getGoalId() != null && STRATEGY_TYPES_WITH_LEARNING_STATUS.contains(strategyData.getName());
    }

    public static boolean isStrategyAlreadyStartedAndNotFinished(StrategyData strategyData, LocalDate now) {
        return strategyData.getStart() != null &&
                strategyData.getFinish() != null &&
                !now.isBefore(strategyData.getStart()) &&
                !now.isAfter(strategyData.getFinish());
    }

    public static BigDecimal getAutobudgetPayForConversionAvgCpaWarning(Set<String> enabledFeatures,
                                                                        Currency currency) {
        boolean isTripledCpaUpperLimitEnabled = enabledFeatures
                .contains(INCREASED_CPA_LIMIT_FOR_PAY_FOR_CONVERSION.getName());

        return isTripledCpaUpperLimitEnabled ?
                currency.getAutobudgetPayForConversionAvgCpaWarningIncreased() :
                currency.getAutobudgetPayForConversionAvgCpaWarning();
    }

    public static ru.yandex.direct.core.entity.campaign.model.StrategyName toStrategyName(ru.yandex.direct.core.entity.strategy.model.StrategyName type) {
        return ru.yandex.direct.core.entity.campaign.model.StrategyName.valueOf(type.name());
    }
}
