package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.unableToDelete;
import static ru.yandex.direct.validation.result.ValidationResult.getInvalidItems;

@Service
@ParametersAreNonnullByDefault
public class DeleteCampaignValidationService {

    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final BsExportQueueRepository bsExportQueueRepository;
    private final PricePackageRepository pricePackageRepository;

    @Autowired
    public DeleteCampaignValidationService(CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                           CampaignRepository campaignRepository,
                                           CampaignTypedRepository campaignTypedRepository,
                                           BsExportQueueRepository bsExportQueueRepository,
                                           PricePackageRepository pricePackageRepository) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.bsExportQueueRepository = bsExportQueueRepository;
        this.pricePackageRepository = pricePackageRepository;
    }

    public ValidationResult<List<Long>, Defect> validate(List<Long> ids, long operatorUid, ClientId clientId,
                                                         int shard) {
        CampaignSubObjectAccessValidator accessValidator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, ids)
                .createValidator(CampaignAccessType.READ_WRITE);

        Set<Long> writableCampaignIds = StreamEx.of(ids)
                .map(accessValidator)
                .filter(vr -> !vr.hasAnyErrors())
                .map(ValidationResult::getValue)
                .toSet();

        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, writableCampaignIds);
        Map<Long, Campaign> campaignById = listToMap(campaigns, Campaign::getId);

        Set<Long> cpmPriceCampaignIds = filterAndMapToSet(campaigns,
                campaign -> campaign.getType() == CampaignType.CPM_PRICE,
                Campaign::getId);
        @SuppressWarnings("unchecked")
        List<CpmPriceCampaign> cpmPriceCampaigns = (List<CpmPriceCampaign>)
                campaignTypedRepository.getTypedCampaigns(shard, cpmPriceCampaignIds);

        List<Long> pricePackageIds = mapList(cpmPriceCampaigns, CpmPriceCampaign::getPricePackageId);
        Map<Long, PricePackage> pricePackagesById = pricePackageRepository.getPricePackages(pricePackageIds);
        Set<Long> autoApprovedPricePackageIds = filterAndMapToSet(pricePackagesById.values(),
                PricePackage::getCampaignAutoApprove,
                PricePackage::getId);

        Set<Long> manuallyApprovedCpmPriceCampaignIds = filterAndMapToSet(cpmPriceCampaigns,
                cpmPriceCampaign -> cpmPriceCampaign.getFlightStatusApprove() == PriceFlightStatusApprove.YES &&
                        !autoApprovedPricePackageIds.contains(cpmPriceCampaign.getPricePackageId()),
                CpmPriceCampaign::getId);

        Set<Long> campaignIdsWithStatusEmptyYes = filterAndMapToSet(campaigns,
                Campaign::getStatusEmpty,
                Campaign::getId);

        Set<Long> campaignsInBsQueue = bsExportQueueRepository.getCampaignsIdsInQueue(shard, ids);

        Map<Long, Long> masterIdsBySubCampaignIds =
                campaignRepository.getSubCampaignIdsWithMasterIds(shard, ids);
        List<Long> subCampaignIds = new ArrayList<>(masterIdsBySubCampaignIds.keySet());
        Set<Long> badMasterCampaignIds = Set.of();

        if (!subCampaignIds.isEmpty()) {
            ValidationResult<List<Long>, Defect> subCampaignIdsValidationResult =
                    validate(subCampaignIds, operatorUid, clientId, shard);
            Set<Long> badSubCampaignIds = getInvalidItems(subCampaignIdsValidationResult);
            badMasterCampaignIds = EntryStream.of(masterIdsBySubCampaignIds)
                    .filterKeys(badSubCampaignIds::contains)
                    .values()
                    .toSet();
        }

        return ListValidationBuilder.of(ids, Defect.class)
                .checkEachBy(accessValidator)
                .checkEach(campaignNotDeleted(campaignIdsWithStatusEmptyYes), When.isValid())
                .checkEach(fromPredicate(
                        id -> isDeletableCampaign(campaignById.get(id), manuallyApprovedCpmPriceCampaignIds),
                        unableToDelete()), When.isValid())
                .checkEach(notInCampaignInBsQueue(campaignsInBsQueue), When.isValid())
                .checkEach(subCampaignsAreOk(badMasterCampaignIds))
                .getResult();
    }

    /**
     * Менять сихнронно с методом
     * {@link ru.yandex.direct.grid.processing.service.campaign.CampaignAccessService#isDeletableCampaign}
     * и методом perl/protected/Direct/Validation/Campaigns.pm::validate_delete_campaigns.
     * Нахождение кампаний в очереди на отправку в БК проверяется отдельно - notInCampaignInBsQueue
     */
    private static boolean isDeletableCampaign(Campaign campaign, Set<Long> manuallyApprovedCpmPriceCampaignIds) {
        return NumberUtils.isZero(campaign.getSum())
                && NumberUtils.isZero(campaign.getSumToPay())
                && NumberUtils.isZero(campaign.getSumLast())
                && campaign.getOrderId() == 0L
                && !(campaign.getCurrencyConverted() && campaign.getCurrency() == CurrencyCode.YND_FIXED)
                && !manuallyApprovedCpmPriceCampaignIds.contains(campaign.getId());
    }

    private static Constraint<Long, Defect> campaignNotDeleted(Set<Long> campaignIdsWithStatusEmptyYes) {
        return fromPredicate(Predicates.notInSet(campaignIdsWithStatusEmptyYes), campaignNotFound());
    }

    private static Constraint<Long, Defect> notInCampaignInBsQueue(Set<Long> campaignsInBsQueue) {
        return fromPredicate(Predicates.notInSet(campaignsInBsQueue), unableToDelete());
    }

    /**
     * Подлежащие кампании недоступны для управления пользователем напрямую - все действия с мастер-кампаниями
     * реплицируются на соответствующие им подлежащие. Мы должны удалять подлежащие кампании вместе с удалением их
     * мастер-кампаний, и не должно быть возможно удалить мастер-кампании при невозможности удалить их подлежащие.
     */
    private static Constraint<Long, Defect> subCampaignsAreOk(Set<Long> badMasterCampaignIds) {
        return fromPredicate(Predicates.notInSet(badMasterCampaignIds), unableToDelete());
    }
}
