package ru.yandex.direct.core.entity.campaign.service.validation;

import java.net.IDN;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.InternetDomainName;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.repository.ClientLimitsRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;

import static ru.yandex.direct.core.entity.client.Constants.DEFAULT_VIDEO_BLACKLIST_SIZE_LIMIT;

@Service
@ParametersAreNonnullByDefault
public class DisableDomainValidationService {
    private final ClientLimitsRepository clientLimitsRepository;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;


    private static final String YANDEX_RF = "xn--d1acpjx3f.xn--p1ai";

    private static final Pattern YANDEX_DOMAINS = Pattern.compile("^((m|www)\\.)?((direct)\\.)?(yandex|ya)\\.[a-z]+$");
    private static final Pattern COMMON_SECOND_LEVEL_DOMAINS =
            Pattern.compile("^(\\*\\.)?((pp|ac|boom|msk|spb|nnov)\\.ru|(net|org|com|int|edu)\\.[a-z]{2})$");

    private static final Pattern APP_ID =
            Pattern.compile("^[A-Za-z][A-Za-z0-9_\\-]*(?:\\.[A-Za-z0-9][A-Za-z0-9_\\-]*)+$");
    private static final Pattern SHORT_ID =
            Pattern.compile("^[A-Za-z0-9][A-Za-z0-9_\\-]*$");
    private static final Pattern NUMBER_ID =
            Pattern.compile("^[1-9][0-9]*$");

    private static final List<Pattern> PATTERN_LIST_WITH_DISABLE_ANY_DOMAINS_ALLOWED =
            Collections.singletonList(COMMON_SECOND_LEVEL_DOMAINS);

    private static final List<Pattern> PATTERN_LIST_WITHOUT_DISABLE_ANY_DOMAINS_ALLOWED =
            Arrays.asList(YANDEX_DOMAINS, COMMON_SECOND_LEVEL_DOMAINS);

    @Autowired
    public DisableDomainValidationService(ClientLimitsRepository clientLimitsRepository, ShardHelper shardHelper,
                                          FeatureService featureService) {
        this.clientLimitsRepository = clientLimitsRepository;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
    }

    public boolean isValidSize(ClientId clientId, int limit) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Collection<ClientLimits> clientLimits =
                clientLimitsRepository.fetchByClientIds(shard, Collections.singletonList(clientId.asLong()));
        if (clientLimits.isEmpty()) {
            return limit <= DEFAULT_VIDEO_BLACKLIST_SIZE_LIMIT;
        }

        ClientLimits clientLimit = clientLimits.iterator().next();
        return limit <= clientLimit.getVideoBlacklistSizeLimitOrDefault();
    }

    public boolean isValidDomains(Collection<String> domains, ClientId clientId) {
        return domains.stream().allMatch(domain -> isValidDomain(domain, clientId));
    }

    public boolean isValidDomain(String domain, ClientId clientId) {
        String punyCode;
        //Не можем быть уверены, что здесь что-то вообще разумное пришло
        try {
            punyCode = IDN.toASCII(domain);
        } catch (IllegalArgumentException e) {
            return false;
        }
        String lower = punyCode.toLowerCase();

        boolean disableAnyDomainsAllowed = featureService.isEnabledForClientId(clientId,
                FeatureName.DISABLE_ANY_DOMAINS_ALLOWED);

        boolean disableNumberIdAndShortBundleIdAllowed = featureService.isEnabledForClientId(clientId,
                FeatureName.DISABLE_NUMBER_ID_AND_SHORT_BUNDLE_ID_ALLOWED);

        List<Pattern> patternList = disableAnyDomainsAllowed
                ? PATTERN_LIST_WITH_DISABLE_ANY_DOMAINS_ALLOWED
                : PATTERN_LIST_WITHOUT_DISABLE_ANY_DOMAINS_ALLOWED;

        for (Pattern p : patternList) {
            Matcher matcher = p.matcher(lower);
            if (matcher.matches()) {
                return false;
            }
        }

        if (YANDEX_RF.equals(lower) && !disableAnyDomainsAllowed) {
            return false;
        }

        boolean res = isValidAppId(lower, disableNumberIdAndShortBundleIdAllowed);
        if (!res) {
            //InternetDomainName.isValid проверяет так же, за исключением того, что при
            //InternetDomainName.from(domain).parts().size() == 1 возвращает true.
            //Поэтому сначала проверяем на appId, а потом на домен
            try {
                return InternetDomainName.from(domain).parts().size() > 1;
            } catch (IllegalArgumentException e) {
                return false;
            }
        }
        return true;
    }

    static boolean isValidAppId(String domain, boolean disableNumberIdAndShortBundleIdAllowed) {
        if (disableNumberIdAndShortBundleIdAllowed) {
            if (SHORT_ID.matcher(domain).matches()) {
                return true;
            }
            if (NUMBER_ID.matcher(domain).matches()) {
                return true;
            }
        }

        Matcher matcher = APP_ID.matcher(domain);
        return matcher.matches();
    }

}
