package ru.yandex.direct.core.entity.campaign.service.validation;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;

import static ru.yandex.direct.core.validation.ValidationUtils.areModelPropertyValuesEqual;
import static ru.yandex.direct.core.validation.defects.RightsDefects.forbiddenToChange;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

/**
 * Упрощалка валидации modelChanges, можно использовать на этапе когда есть загруженные модели.
 * <p>
 * Для данной модели хранит список пар ModelProperty, PropertyChangePermission и позволяет прогнать проверку
 * всего списка, чтобы найти те property, которые поменялись, но не должны меняться и добавляет на них дефект.
 * <p>
 * Этот класс может быть предпочтительнее классического Constraint по следующим причинам:
 * <p>
 * 1. Не нужно самостоятельно каждый раз проверять, что новое значение отличается от старого, эта проверка всегда
 * выполняется (а для BigDecimal сравнение идёт через compareTo, а не equals, что легко забыть, если не использовать
 * этот класс)
 * <p>
 * 2. В методе PropertyChangePermission который требуется реализовать отсутствует Defect и это удобно по сравнение с
 * Constraint-ом, поскольку Defect обычно меняется значительно реже условия при котором этот дефект нужно навесить.
 * А удачно подобранное название метода уже в достаточной степени описывает какой примерно дефект будет навешен.
 * Отсутствие дефекта позволяет избежать многочисленных конструкций "Constraint.fromPredicate".
 * <p>
 * T - тип модели, например CpmPricePackage
 * C - контекст, может содержать например роль оператора и/или клиента и т.п.
 * V - тип проперти которую меняем, например Date
 * modelProperty - пропертя, которую меняем, например CpmPricePackage.START_DATE
 */
@ParametersAreNonnullByDefault
public class PropertyChangeValidator<T extends ModelWithId, C> {

    private final ImmutableList<PermissionTest<T, C, ?>> permissions;

    private PropertyChangeValidator(ImmutableList<PermissionTest<T, C, ?>> permissions) {
        this.permissions = permissions;
    }

    public static <T extends ModelWithId, C> PropertyChangeValidator.Builder<T, C> newBuilder() {
        return new Builder<>();
    }

    public ValidationResult<ModelChanges<T>, Defect> validate(ModelChanges<T> modelChanges,
                                                              @Nullable T model,
                                                              C context) {
        if (model == null) {
            return ValidationResult.success(modelChanges);
        }

        var vb = ModelChangesValidationBuilder.of(modelChanges);
        permissions.forEach(permissionTest -> permissionTest
                .addDefectOnFieldIfChangedButForbidden(model, context, vb));
        return vb.getResult();
    }

    public static class Builder<T extends ModelWithId, C> {

        private ImmutableList.Builder<PermissionTest<T, C, ?>> builder = ImmutableList.builder();

        /**
         * @see PropertyChangeValidator.Builder#add(ModelProperty, PropertyChangePermission, Defect)
         */
        public <V> Builder<T, C> add(ModelProperty<? super T, V> modelProperty,
                                     PropertyChangePermission<T, C, V> permission) {
            return add(modelProperty, permission, forbiddenToChange());
        }

        /**
         * Добавляет проверку.
         * <p>Можно добавлять несколько проверок на одну и ту же property - все будут исполнены
         * <p>Свободно можно использовать один и тот же permission для разных пропертей - но тогда
         * {@link PropertyChangePermission#canChangeProperty(Model, Object, Object)} вызовется столько раз, сколько он
         * был использован. В будущем это возможно будет сооптимизировано.
         */
        public <V> Builder<T, C> add(ModelProperty<? super T, V> modelProperty,
                                     PropertyChangePermission<T, C, V> permission,
                                     Defect<?> defect) {
            builder.add(new PermissionTest<>(modelProperty, permission, defect));
            return this;
        }

        public PropertyChangeValidator<T, C> build() {
            return new PropertyChangeValidator<>(builder.build());
        }

    }

    private static class PermissionTest<T extends ModelWithId, C, V> {

        private final ModelProperty<? super T, V> modelProperty;
        private final PropertyChangePermission<T, C, V> permission;
        private final Defect<?> defect;

        PermissionTest(ModelProperty<? super T, V> modelProperty,
                       PropertyChangePermission<T, C, V> permission,
                       Defect<?> defect) {
            this.modelProperty = modelProperty;
            this.permission = permission;
            this.defect = defect;
        }

        void addDefectOnFieldIfChangedButForbidden(T model,
                                                   C context,
                                                   ModelChangesValidationBuilder<T> vb) {
            vb.item(modelProperty)
                    .check(fromPredicate(newPropertyValue -> {
                        var oldPropertyValue = modelProperty.get(model);
                        return areModelPropertyValuesEqual(oldPropertyValue, newPropertyValue) ||
                                permission.canChangeProperty(model, context, newPropertyValue);
                        },
                            defect));
        }
    }

}
