package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignAlreadySuspended;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotSuspended;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class SuspendResumeCampaignValidationService {
    private static final int MAX_ELEMENTS_PER_OPERATION = 1000;

    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ModelChangesValidationTool preValidationTool;
    private final FeatureService featureService;

    @Autowired
    public SuspendResumeCampaignValidationService(
            CampaignRepository campaignRepository,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            FeatureService featureService
    ) {
        this.campaignRepository = campaignRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.preValidationTool = ModelChangesValidationTool.builder()
                .objectNotFoundDefect(campaignNotFound())
                .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION).build();
        this.featureService = featureService;
    }

    public ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChanges(
            List<ModelChanges<Campaign>> modelChangesList,
            long operatorUid, ClientId clientId, int shard
    ) {
        List<Long> campaignIds = mapList(modelChangesList, ModelChanges::getId);
        Set<Long> existingCampaignIds = campaignRepository.getExistingCampaignIds(shard, clientId, campaignIds);

        var accessValidator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, campaignIds)
                .createValidator(CampaignAccessType.READ_WRITE);

        ValidationResult<List<ModelChanges<Campaign>>, Defect> validationResult =
                preValidationTool.validateModelChangesList(modelChangesList, existingCampaignIds);

        return new ListValidationBuilder<>(validationResult)
                .checkEachBy(c -> campaignAccessValidator(c, accessValidator), When.isValid())
                .getResult();
    }

    private ValidationResult<ModelChanges<Campaign>, Defect> campaignAccessValidator(
            ModelChanges<Campaign> changes,
            CampaignSubObjectAccessValidator accessValidator) {
        ItemValidationBuilder<ModelChanges<Campaign>, Defect> ivb = ItemValidationBuilder.of(changes);
        ivb.item(changes.getId(), Campaign.ID.name()).checkBy(accessValidator, When.isValid());
        return ivb.getResult();
    }

    public ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<Campaign>>, Defect> preValidateResult,
            Map<Long, Campaign> campaigns, boolean resume
    ) {
        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(isNotEmpty(campaigns), When.isValid())
                .checkEach(isNotArchived(campaigns), When.isValidAnd(When.isTrue(resume)))
                .weakCheckEach(isSuspended(campaigns), When.isValidAnd(When.isTrue(resume)))
                .weakCheckEach(isNotSuspended(campaigns), When.isValidAnd(When.isFalse(resume)))
                .getResult();
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNotEmpty(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) && !campaigns.get(mc.getId()).getStatusEmpty(),
                campaignNotFound());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNotArchived(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) && !campaigns.get(mc.getId()).getStatusArchived(),
                archivedCampaignModification());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isNotSuspended(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) && campaigns.get(mc.getId()).getStatusShow(),
                campaignAlreadySuspended());
    }

    private Constraint<ModelChanges<Campaign>, Defect> isSuspended(Map<Long, Campaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.containsKey(mc.getId()) && !campaigns.get(mc.getId()).getStatusShow(),
                campaignNotSuspended());
    }
}
