package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignIsInSpecialArchivedState;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotArchived;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class UnarchiveCampaignValidationService {
    private static final int MAX_ELEMENTS_PER_OPERATION = 1000;

    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final CampaignLimitsValidationService campaignLimitsValidationService;
    private final FeatureService featureService;
    private final ClientService clientService;
    private final AdGroupRepository adGroupRepository;
    private final ModelChangesValidationTool preValidationTool;

    @Autowired
    public UnarchiveCampaignValidationService(
            CampaignRepository campaignRepository,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            CampaignLimitsValidationService campaignLimitsValidationService,
            FeatureService featureService,
            ClientService clientService,
            AdGroupRepository adGroupRepository
    ) {
        this.campaignRepository = campaignRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.campaignLimitsValidationService = campaignLimitsValidationService;
        this.featureService = featureService;
        this.clientService = clientService;
        this.adGroupRepository = adGroupRepository;
        this.preValidationTool = ModelChangesValidationTool.builder()
                .objectNotFoundDefect(campaignNotFound())
                .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION).build();
    }

    public ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validateModelChanges(
            List<ModelChanges<BaseCampaign>> modelChangesList,
            long operatorUid, ClientId clientId, int shard
    ) {
        List<Long> campaignIds = mapList(modelChangesList, ModelChanges::getId);
        Set<Long> existingCampaignIds = new HashSet<>(campaignRepository.getExistingNotEmptyCampaignIds(
                shard, clientId, campaignIds));

        CampaignSubObjectAccessValidator accessValidator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, existingCampaignIds)
                .createValidator(CampaignAccessType.READ_WRITE);

        ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validationResult =
                preValidationTool.validateModelChangesList(modelChangesList, existingCampaignIds);

        return new ListValidationBuilder<>(validationResult)
                .checkEachBy(c -> campaignAccessValidator(c, accessValidator), When.isValid())
                .getResult()
                .transformUnchecked(new Transformer());
    }

    private ValidationResult<ModelChanges<BaseCampaign>, Defect> campaignAccessValidator(
            ModelChanges<BaseCampaign> changes,
            CampaignSubObjectAccessValidator accessValidator) {
        ItemValidationBuilder<ModelChanges<BaseCampaign>, Defect> ivb =
                ItemValidationBuilder.of(changes);
        ivb.item(changes.getId(), BaseCampaign.ID.name())
                .checkBy(accessValidator, When.isValid());
        return ivb.getResult();
    }

    public ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validateModelChangesBeforeApply(
            int shard, ClientId clientId,
            ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> preValidateResult,
            Map<Long, BaseCampaign> campaigns
    ) {
        Client client = clientService.getClient(clientId);
        Set<String> availableFeatures = featureService.getEnabledForClientId(clientId);
        Map<Long, GeoproductAvailability> isGeoByCampaignId =
                adGroupRepository.getGeoproductAvailabilityByCampaignId(shard, campaigns.keySet());
        Map<Long, ContentPromotionAdgroupType> contentPromotionTypeByCampaignId =
                adGroupRepository.getContentPromotionAdGroupTypeByCampaignId(shard, campaigns.keySet());
        boolean cpmGeoproductEnabled = availableFeatures.contains(FeatureName.CPM_GEOPRODUCT_ENABLED.getName());
        boolean cpmDealsEnabled = availableFeatures.contains(FeatureName.CPM_DEALS.getName());
        boolean contentPromotionVideoEnabled = availableFeatures.contains(
                FeatureName.CONTENT_PROMOTION_VIDEO.getName());
        boolean contentPromotionCollectionEnabled = availableFeatures.contains(
                FeatureName.CONTENT_PROMOTION_COLLECTION.getName());

        ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> vr =
                new ListValidationBuilder<>(preValidateResult)
                        .checkEach(isNotGeoOrFeatureEnabled(cpmGeoproductEnabled, isGeoByCampaignId), When.isValid())
                        .checkEach(isNotCpmDealsOrFeatureEnabled(cpmDealsEnabled, campaigns), When.isValid())
                        .checkEach(isNotContentPromotionOrFeatureEnabled(contentPromotionVideoEnabled,
                                        contentPromotionCollectionEnabled,
                                        contentPromotionTypeByCampaignId,
                                        campaigns),
                                When.isValid())
                        .checkEach(isCurrencyMatched(client, campaigns), When.isValid())
                        .weakCheckEach(isArchived(campaigns), When.isValid())
                        .getResult();
        campaignLimitsValidationService.unarcCampaignsCountLimitCheck(shard, clientId, vr);
        return vr;
    }

    // не даем разархивировать геопродуктовую кампанию без фичи cpm_geoproduct_enabled
    private static Constraint<ModelChanges<BaseCampaign>, Defect> isNotGeoOrFeatureEnabled(
            boolean cpmGeoproductEnabled, Map<Long, GeoproductAvailability> isGeoByCampaignId) {
        return Constraint.fromPredicate(
                mc -> cpmGeoproductEnabled || isGeoByCampaignId.get(mc.getId()) != GeoproductAvailability.YES,
                campaignIsInSpecialArchivedState());
    }

    // не даем разархивировать охватные продукты без фичи cpm_deals
    private static Constraint<ModelChanges<BaseCampaign>, Defect> isNotCpmDealsOrFeatureEnabled(
            boolean cpmDealsEnabled, Map<Long, BaseCampaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> cpmDealsEnabled ||
                        ((CampaignWithCampaignType) campaigns.get(mc.getId())).getType() != CampaignType.CPM_DEALS,
                campaignIsInSpecialArchivedState());
    }

    // не даем разархивировать продвижение контента без соответствующих фич
    private static Constraint<ModelChanges<BaseCampaign>, Defect> isNotContentPromotionOrFeatureEnabled(
            boolean contentPromotionVideoEnabled,
            boolean contentPromotionCollectionEnabled,
            Map<Long, ContentPromotionAdgroupType> contentPromotionTypeByCampaignId,
            Map<Long, BaseCampaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> {
                    if (((CampaignWithCampaignType) campaigns.get(mc.getId())).getType()
                            != CampaignType.CONTENT_PROMOTION) {
                        return true;
                    }
                    switch (contentPromotionTypeByCampaignId.get(mc.getId())) {
                        case VIDEO:
                            return contentPromotionVideoEnabled;
                        case COLLECTION:
                            return contentPromotionCollectionEnabled;
                        case SERVICE:
                        case EDA:
                        default:
                            return true;
                    }
                },
                campaignIsInSpecialArchivedState());
    }

    // валюта кампании совпадает с валютой клиента
    private Constraint<ModelChanges<BaseCampaign>, Defect> isCurrencyMatched(Client client, Map<Long,
            BaseCampaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.get(mc.getId()) instanceof CommonCampaign
                        && ((CommonCampaign) campaigns.get(mc.getId())).getCurrency() == client.getWorkCurrency(),
                campaignIsInSpecialArchivedState());
    }

    private Constraint<ModelChanges<BaseCampaign>, Defect> isArchived(Map<Long, BaseCampaign> campaigns) {
        return Constraint.fromPredicate(
                mc -> campaigns.get(mc.getId()) instanceof CommonCampaign
                        && ((CommonCampaign) campaigns.get(mc.getId())).getStatusArchived(),
                campaignNotArchived());
    }

    @ParametersAreNonnullByDefault
    private static class Transformer implements ValidationResult.ValidationResultTransformer<Defect> {
        @Override
        public List<Defect> transformErrors(@SuppressWarnings("unused") Path path, List<Defect> errors) {
            return filterList(errors,
                    error -> !error.defectId().equals(CampaignDefectIds.Gen.ARCHIVED_CAMPAIGN_MODIFICATION));
        }
    }
}
