package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNoRights;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.minusKeywordsNotAllowed;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseBeforeNormalizationValidator.minusKeywordsAreValidBeforeNormalization;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.maxLengthKeywordsWithoutSpecSymbolsAndSpaces;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.maxWordsInKeywords;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Service
@ParametersAreNonnullByDefault
public class UpdateCampaignValidationService {
    private static final int MAX_ELEMENTS_PER_OPERATION = 1000;

    public static final int MAX_NAME_LENGTH = 255;

    public static final List<CampaignType> CAMPAIGN_TYPES_WITHOUT_MINUS_KEYWORDS =
            ImmutableList.of(CampaignType.CPM_YNDX_FRONTPAGE);

    private final RbacService rbacService;
    private final CampaignRepository campaignRepository;
    private final ModelChangesValidationTool preValidationTool;

    @Autowired
    public UpdateCampaignValidationService(RbacService rbacService, CampaignRepository campaignRepository) {
        this.rbacService = rbacService;
        this.campaignRepository = campaignRepository;
        this.preValidationTool = ModelChangesValidationTool.builder()
                .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION).build();
    }

    public ValidationResult<List<ModelChanges<Campaign>>, Defect> preValidate(
            List<ModelChanges<Campaign>> modelChangesList,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            long operatorUid, ClientId clientId, int shard) {
        List<Long> campaignIds = mapList(modelChangesList, ModelChanges::getId);
        Set<Long> existingCampaignIds = campaignRepository.getExistingCampaignIds(shard, clientId, campaignIds);
        Set<Long> visibleCampaigns = rbacService.getVisibleCampaigns(operatorUid, existingCampaignIds);
        Set<Long> writableCampaigns = rbacService.getWritableCampaigns(operatorUid, existingCampaignIds);

        ValidationResult<List<ModelChanges<Campaign>>, Defect> validationResult =
                preValidationTool.validateModelChangesList(modelChangesList, existingCampaignIds);

        return new ListValidationBuilder<>(validationResult)
                .checkEachBy(campaignAccessValidator(visibleCampaigns, writableCampaigns), When.isValid())
                .checkEachBy(minusKeywordsBeforeNormalizationValidator(minusPhraseValidationMode), When.isValid())
                .getResult();
    }

    private Validator<ModelChanges<Campaign>, Defect> campaignAccessValidator(Set<Long> visibleCampaigns,
                                                                              Set<Long> writableCampaigns) {
        return changes -> {
            ItemValidationBuilder<ModelChanges<Campaign>, Defect> ivb = ItemValidationBuilder.of(changes);

            ivb.item(changes.getId(), Campaign.ID.name())
                    .check(inSet(visibleCampaigns), campaignNotFound(), When.isValid())
                    .check(inSet(writableCampaigns), campaignNoRights(), When.isValid());

            return ivb.getResult();
        };
    }

    public ValidationResult<List<Campaign>, Defect> validate(
            ValidationResult<List<Campaign>, Defect> preValidationResult) {
        return new ListValidationBuilder<>(preValidationResult)
                .checkEachBy(this::validateCampaign, When.isValid())
                .getResult();
    }

    private Validator<ModelChanges<Campaign>, Defect> minusKeywordsBeforeNormalizationValidator(
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode) {
        return modelChanges -> {
            ItemValidationBuilder<ModelChanges<Campaign>, Defect> vb =
                    ItemValidationBuilder.of(modelChanges, Defect.class);

            if (modelChanges.isPropChanged(Campaign.MINUS_KEYWORDS)
                    && modelChanges.getChangedProp(Campaign.MINUS_KEYWORDS) != null) {
                vb.item(modelChanges.getChangedProp(Campaign.MINUS_KEYWORDS), Campaign.MINUS_KEYWORDS.name())
                        .checkBy(minusKeywordsAreValidBeforeNormalization(
                                CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION,
                                minusPhraseValidationMode));
            }

            return vb.getResult();
        };
    }

    private ValidationResult<Campaign, Defect> validateCampaign(Campaign campaign) {
        ModelItemValidationBuilder<Campaign> vb = ModelItemValidationBuilder.of(campaign);

        vb.item(Campaign.NAME)
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(MAX_NAME_LENGTH));

        vb.item(Campaign.STATUS_ARCHIVED).check(fromPredicate(archived -> !archived, archivedCampaignModification()));
        vb.item(Campaign.MINUS_KEYWORDS)
                .checkBy(minusKeywordsWithCampaignTypeValidator(campaign.getType()),
                        When.valueIs(t -> t != null && !t.isEmpty()))
                .check(maxLengthKeywordsWithoutSpecSymbolsAndSpaces(CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH))
                .check(maxWordsInKeywords());

        return vb.getResult();
    }

    private Validator<List<String>, Defect> minusKeywordsWithCampaignTypeValidator(CampaignType campaignType) {
        return keywords -> {
            if (keywords != null && !keywords.isEmpty() &&
                    CAMPAIGN_TYPES_WITHOUT_MINUS_KEYWORDS.contains(campaignType)) {
                return ValidationResult.failed(keywords, minusKeywordsNotAllowed());
            }
            return ValidationResult.success(keywords);
        };
    }

    public boolean isCampaignWritable(long operatorUid, long campaignId) {
        return rbacService.getWritableCampaigns(operatorUid, Collections.singletonList(campaignId)).contains(campaignId);
    }
}
