package ru.yandex.direct.core.entity.campaign.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.type.update.CampaignUpdateValidationTypeSupportFacade;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.inconsistentCampaignType;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterValidSubResults;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
@ParametersAreNonnullByDefault
public class UpdateRestrictedCampaignValidationService {

    private final FeatureService featureService;
    private final CampaignUpdateValidationTypeSupportFacade typeSupportFacade;
    private final ModelChangesValidationTool updateValidationTool;
    private final CampaignSubObjectAccessCheckerFactory accessCheckerFactory;

    @Autowired
    public UpdateRestrictedCampaignValidationService(FeatureService featureService,
                                                     CampaignUpdateValidationTypeSupportFacade typeSupportFacade,
                                                     CampaignSubObjectAccessCheckerFactory accessCheckerFactory) {
        this.featureService = featureService;
        this.typeSupportFacade = typeSupportFacade;
        this.accessCheckerFactory = accessCheckerFactory;
        this.updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    public ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> preValidate(
            RestrictedCampaignsUpdateOperationContainer container,
            List<? extends ModelChanges<? extends BaseCampaign>> modelChanges,
            Set<Long> clientCampaignIds) {

        List<ModelChanges<BaseCampaign>> campaignModelChanges = mapList(modelChanges,
                mc -> mc.castModelUp(BaseCampaign.class));

        CampaignSubObjectAccessChecker accessChecker =
                accessCheckerFactory.newCampaignChecker(container.getOperatorUid(), container.getClientId(),
                        clientCampaignIds);
        CampaignSubObjectAccessValidator accessCheckerValidator =
                accessChecker.createValidator(CampaignAccessType.READ_WRITE);
        var vr = validateOperatorAccess(campaignModelChanges, accessCheckerValidator);

        updateValidationTool.validateModelChangesList(vr, () -> clientCampaignIds);

        new ListValidationBuilder<>(vr)
                .checkEach(modelChangesHasConsistentType(container), When.isValid());

        var vrWithValidSubResults = filterValidSubResults(vr);
        typeSupportFacade.preValidate(container, vrWithValidSubResults);
        return vr;
    }

    private Constraint<ModelChanges<BaseCampaign>, Defect> modelChangesHasConsistentType(
            RestrictedCampaignsUpdateOperationContainer container) {
        return fromPredicate(mc -> {
            Class<? extends BaseCampaign> modelChangesClass = mc.getModelType();
            Class<? extends BaseCampaign> runtimeClass = container.getRuntimeClass(mc.getId());
            return runtimeClass != null && modelChangesClass.isAssignableFrom(runtimeClass);
        }, inconsistentCampaignType());
    }

    public ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validateBeforeApply(
            CampaignValidationContainer container,
            ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> vr,
            Map<Long, BaseCampaign> unmodifiedModels) {
        typeSupportFacade.validateBeforeApply(container, vr, unmodifiedModels);
        return vr;
    }

    public ValidationResult<? extends List<? extends BaseCampaign>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<? extends List<? extends BaseCampaign>, Defect> vr,
            Map<Integer, ? extends AppliedChanges<? extends BaseCampaign>> appliedChangesForValidModelChanges) {

        typeSupportFacade.validate(container, vr, appliedChangesForValidModelChanges);
        return vr;
    }

    private ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validateOperatorAccess(
            List<ModelChanges<BaseCampaign>> modelChanges, CampaignSubObjectAccessValidator accessCheckerValidator) {
        ListValidationBuilder<ModelChanges<BaseCampaign>, Defect> vb = ListValidationBuilder.of(modelChanges);

        vb.checkEachBy(mc -> validateOperatorAccessSingleItem(mc, accessCheckerValidator));

        return vb.getResult();
    }

    private <T extends BaseCampaign> ValidationResult<ModelChanges<T>, Defect> validateOperatorAccessSingleItem(
            ModelChanges<T> modelChanges, CampaignSubObjectAccessValidator accessCheckerValidator) {

        ItemValidationBuilder<ModelChanges<T>, Defect> vb = ItemValidationBuilder.of(modelChanges);

        vb.item(modelChanges.getId(), "id")
                .check(validId(), When.notNull())
                .checkBy(accessCheckerValidator, When.isValidAnd(When.notNull()));

        return vb.getResult();
    }
}
