package ru.yandex.direct.core.entity.campaign.service.validation.type;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@ParametersAreNonnullByDefault
public class TimeIntervalValidator implements Validator<TimeInterval, Defect> {

    public static final Integer MIN_HOUR = 0;
    static final Integer MAX_HOUR = 24;
    static final Integer MIN_MINUTE = 0;
    static final Integer MAX_MINUTE = 59;

    @Override
    public ValidationResult<TimeInterval, Defect> apply(TimeInterval timeInterval) {
        var vb = ModelItemValidationBuilder.of(timeInterval);
        vb.item(TimeInterval.START_HOUR)
                .check(notNull())
                .check(inRange(MIN_HOUR, MAX_HOUR), When.notNull());
        vb.item(TimeInterval.START_MINUTE)
                .check(notNull())
                .check(inRange(MIN_MINUTE, MAX_MINUTE), When.notNull())
                .check(validMinute(), When.isValid())
                .check(zeroMinuteIfEndOfDay(), When.isTrue(MAX_HOUR.equals(timeInterval.getStartHour())));

        vb.item(TimeInterval.END_HOUR)
                .check(notNull())
                .check(inRange(MIN_HOUR, MAX_HOUR), When.notNull());
        vb.item(TimeInterval.END_MINUTE)
                .check(notNull())
                .check(inRange(MIN_MINUTE, MAX_MINUTE), When.notNull())
                .check(validMinute(), When.isValid())
                .check(zeroMinuteIfEndOfDay(), When.isTrue(MAX_HOUR.equals(timeInterval.getEndHour())));

        return vb.getResult();
    }

    /**
     * Проверяет, что минута равна 0, если {@link TimeInterval#END_HOUR} равен {@link #MAX_HOUR}
     */
    private static Constraint<Integer, Defect> zeroMinuteIfEndOfDay() {
        return fromPredicate(minute -> minute == 0, CommonDefects.invalidFormat());
    }

    /**
     * Проверяет, что минуты заданы кратно 15 (0, 15, 30, 45)
     */
    private static Constraint<Integer, Defect> validMinute() {
        return fromPredicate(minute -> minute % 15 == 0, CommonDefects.invalidFormat());
    }

}
