package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithAllowedPageIds;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithAllowedPageIdsValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.operatorCannotSetAllowedPageIds;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.isEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithAllowedPageIdsAddValidationTypeSupport
        extends AbstractCampaignAddValidationTypeSupport<CampaignWithAllowedPageIds> {

    private final FeatureService featureService;
    private final SspPlatformsRepository sspPlatformsRepository;

    @Autowired
    public CampaignWithAllowedPageIdsAddValidationTypeSupport(FeatureService featureService,
                                                              SspPlatformsRepository sspPlatformsRepository) {
        this.featureService = featureService;
        this.sspPlatformsRepository = sspPlatformsRepository;
    }

    @Override
    public Class<CampaignWithAllowedPageIds> getTypeClass() {
        return CampaignWithAllowedPageIds.class;
    }

    @Override
    public ValidationResult<List<CampaignWithAllowedPageIds>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithAllowedPageIds>, Defect> vr) {

        boolean canSetAllowedPageIds = featureService.isEnabledForUid(
                container.getOperatorUid(), List.of(FeatureName.SET_CAMPAIGN_ALLOWED_PAGE_IDS));

        return new ListValidationBuilder<>(vr)
                .checkEachBy(this::preValidateCampaignWithoutFeature, When.isFalse(canSetAllowedPageIds))
                .getResult();
    }

    public ValidationResult<CampaignWithAllowedPageIds, Defect> preValidateCampaignWithoutFeature(
            CampaignWithAllowedPageIds campaign) {
        ModelItemValidationBuilder<CampaignWithAllowedPageIds> vb = ModelItemValidationBuilder.of(campaign);
        vb.item(CampaignWithAllowedPageIds.ALLOWED_PAGE_IDS)
                .check(isNull(), operatorCannotSetAllowedPageIds());

        vb.item(CampaignWithAllowedPageIds.ALLOWED_DOMAINS)
                .check(isNull(), operatorCannotSetAllowedPageIds());

        vb.item(CampaignWithAllowedPageIds.ALLOWED_SSP)
                .check(isEmptyCollection(), operatorCannotSetAllowedPageIds());

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<CampaignWithAllowedPageIds>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithAllowedPageIds>, Defect> vr) {
        return new ListValidationBuilder<>(vr)
                .checkEachBy(new CampaignWithAllowedPageIdsValidator(sspPlatformsRepository))
                .getResult();
    }
}
