package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandLiftExperimentsService;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithBrandLiftValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.type.update.CampaignWithBroadMatchUpdateValidationTypeSupport;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.RetargetingCondition;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.feature.FeatureName.BRAND_LIFT;
import static ru.yandex.direct.metrika.client.model.response.RetargetingCondition.Type.AB_SEGMENT;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notTrue;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBrandLiftAddValidationTypeSupport extends AbstractCampaignAddValidationTypeSupport<CampaignWithBrandLift> {
    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithBroadMatchUpdateValidationTypeSupport.class);

    private final FeatureService featureService;
    private final CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService;

    @Autowired
    public CampaignWithBrandLiftAddValidationTypeSupport(
            FeatureService featureService,
            CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService) {
        this.featureService = featureService;
        this.campaignWithBrandLiftExperimentsService = campaignWithBrandLiftExperimentsService;
    }

    @Override
    public Class<CampaignWithBrandLift> getTypeClass() {
        return CampaignWithBrandLift.class;
    }

    @Override
    public ValidationResult<List<CampaignWithBrandLift>, Defect> preValidate(CampaignValidationContainer container,
                                                                             ValidationResult<List<CampaignWithBrandLift>, Defect> vr) {

        boolean abSegmentsEnabledForClient = featureService.isEnabledForClientId(container.getClientId(),
                FeatureName.AB_SEGMENTS);

        boolean brandLiftEnabledForClient = featureService.isEnabledForClientId(container.getClientId(),
                BRAND_LIFT);

        return new ListValidationBuilder<>(vr)
                .checkEachBy(campaign -> {
                    var vb = ModelItemValidationBuilder.of(campaign);
                    vb.item(CampaignWithBrandLift.BRAND_SURVEY_ID)
                            .check(isNull(), When.isFalse(brandLiftEnabledForClient));
                    vb.item(CampaignWithBrandLift.SECTION_IDS)
                            .check(isNull(), When.isFalse(abSegmentsEnabledForClient));
                    vb.item(CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS)
                            .check(isNull(), When.isFalse(abSegmentsEnabledForClient));
                    vb.item(CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN)
                            .check(notTrue());

                    return vb.getResult();
                })
                .getResult();
    }

    @Override
    public ValidationResult<List<CampaignWithBrandLift>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithBrandLift>, Defect> vr) {

        ClientId clientId = container.getClientId();
        ListValidationBuilder<CampaignWithBrandLift, Defect> vb = new ListValidationBuilder<>(vr);

        // При копировании, аналогично перловому, не проверяем доступность целей для клиента, поэтому получать
        // доступные цели смысла нет.
        if (container.isCopy()) {
            return vr;
        }

        List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> clientAbSegments;
        try {
            Map<Long, List<RetargetingCondition>> goalsByUids = container.getMetrikaClient().getAbSegmentGoals();
            clientAbSegments = getClientAbSegments(goalsByUids);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
            vb.checkEach(metrikaReturnsResultWithErrors(), When.notNull());
            return vb.getResult();
        }

        return new ListValidationBuilder<>(vr)
                .checkEachBy(
                        new CampaignWithBrandLiftValidator(emptyMap(), emptyMap(), clientAbSegments, emptyMap(),
                                emptyMap()),
                        When.isValid())
                .getResult();
    }

    private List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> getClientAbSegments(Map<Long,
            List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> goalsByUids) {
        return EntryStream.of(goalsByUids)
                .values()
                .flatMap(Collection::stream)
                .filter(x -> x.getType() == AB_SEGMENT)
                .toList();
    }
}
