package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.net.InetAddress;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.service.validation.type.MobileContentServiceHelper;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.CampaignWithCustomStrategyValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.StrategyValidatorConstantsBuilder;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toCampaignTypeWithCounterIds;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.ALL_CAMPAIGNS_PLATFORMS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.ALL_CAMP_OPTIONS_STRATEGIES;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_CONTEXT_STRATEGY_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_PLATFORM_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_STRATEGY_TYPE;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithCustomStrategyAddValidationTypeSupport extends AbstractCampaignAddValidationTypeSupport<CampaignWithCustomStrategy> {
    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithCustomStrategyAddValidationTypeSupport.class);

    private final ClientService clientService;
    private final FeatureService featureService;
    private final CampaignGoalsService campaignGoalsService;
    private final NetAcl netAcl;
    private final MobileContentServiceHelper mobileContentServiceHelper;

    @Autowired
    public CampaignWithCustomStrategyAddValidationTypeSupport(ClientService clientService,
                                                              FeatureService featureService,
                                                              CampaignGoalsService campaignGoalsService,
                                                              NetAcl netAcl, MobileAppService mobileAppService,
                                                              MobileContentServiceHelper mobileContentServiceHelper) {
        this.clientService = clientService;
        this.featureService = featureService;
        this.campaignGoalsService = campaignGoalsService;
        this.netAcl = netAcl;
        this.mobileContentServiceHelper = mobileContentServiceHelper;
    }

    @Override
    public Class<CampaignWithCustomStrategy> getTypeClass() {
        return CampaignWithCustomStrategy.class;
    }

    @Override
    public ValidationResult<List<CampaignWithCustomStrategy>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithCustomStrategy>, Defect> vr) {
        var clientId = container.getClientId();
        var operatorUid = container.getOperatorUid();
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);

        var campaigns = listToSet(vr.getValue(),
                campaign -> toCampaignTypeWithCounterIds(campaign, enabledFeatures,
                        container.getGoalIdToCounterIdForCampaignsWithoutCounterIds()));
        var vb = new ListValidationBuilder<>(vr);

        Map<CampaignTypeWithCounterIds, Set<Goal>> availableGoalsByCampaign;
        try {
            availableGoalsByCampaign = campaignGoalsService.getAvailableGoalsForCampaignType(
                    operatorUid,
                    clientId,
                    campaigns,
                    container.getMetrikaClient()
            );
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            vb.checkEach(metrikaReturnsResultWithErrors());
            return vb.getResult();
        }

        Currency currency = clientService.getWorkCurrency(clientId);

        InetAddress secretRemoteAddress = CoreHttpUtil.getRemoteAddressFromAuthOrDefault();
        boolean requestFromInternalNetwork = Optional.ofNullable(secretRemoteAddress)
                .map(netAcl::isInternalIp)
                .orElse(false);

        return vb
                .checkEachBy(campaign -> new CampaignWithCustomStrategyValidator(
                                currency,
                                availableGoalsByCampaign.getOrDefault(
                                        toCampaignTypeWithCounterIds(campaign, enabledFeatures,
                                                container.getGoalIdToCounterIdForCampaignsWithoutCounterIds()
                                        ),
                                        Set.of()),
                                Collections::emptyList,
                                Collections::emptyList,
                                banners -> emptyList(),
                                campaign,
                                CAMPAIGN_TO_STRATEGY_TYPE.getOrDefault(campaign.getType(), emptySet()),
                                CAMPAIGN_TO_CONTEXT_STRATEGY_TYPE.getOrDefault(campaign.getType(),
                                        ALL_CAMP_OPTIONS_STRATEGIES),
                                CAMPAIGN_TO_PLATFORM_TYPE.getOrDefault(campaign.getType(), ALL_CAMPAIGNS_PLATFORMS),
                                StrategyValidatorConstantsBuilder.build(campaign.getType(), currency),
                                enabledFeatures,
                                container,
                                // для создаваемой кампании еще нет групп объявлений, поэтому нет и restrictions на
                                // ставку
                                null,
                                requestFromInternalNetwork,
                                mobileAppIds -> mobileContentServiceHelper.getMobileContents(clientId, mobileAppIds),
                                mobileAppId -> mobileContentServiceHelper.getMobileApp(clientId, mobileAppId),
                                false,
                                null
                        ).apply(campaign)
                )
                .getResult();
    }

}
