package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoalsWithRequiredFields;
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithMeaningfulGoalsValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toCampaignTypeWithCounterIds;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithMeaningfulGoalsAddValidationTypeSupport
        extends AbstractCampaignAddValidationTypeSupport<CampaignWithMeaningfulGoalsWithRequiredFields> {
    private static final Logger logger =
            LoggerFactory.getLogger(CampaignWithMeaningfulGoalsAddValidationTypeSupport.class);

    private final CampaignGoalsService campaignGoalsService;
    private final NetAcl netAcl;
    private final FeatureService featureService;

    @Autowired
    public CampaignWithMeaningfulGoalsAddValidationTypeSupport(CampaignGoalsService campaignGoalsService,
                                                               NetAcl netAcl,
                                                               FeatureService featureService) {
        this.campaignGoalsService = campaignGoalsService;
        this.netAcl = netAcl;
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithMeaningfulGoalsWithRequiredFields> getTypeClass() {
        return CampaignWithMeaningfulGoalsWithRequiredFields.class;
    }

    @Override
    public ValidationResult<List<CampaignWithMeaningfulGoalsWithRequiredFields>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithMeaningfulGoalsWithRequiredFields>, Defect> vr) {
        var clientId = container.getClientId();
        var operatorUid = container.getOperatorUid();
        var vb = new ListValidationBuilder<>(vr);

        List<CampaignWithMeaningfulGoalsWithRequiredFields> campaigns = vr.getValue();

        Map<CampaignWithMeaningfulGoalsWithRequiredFields, Set<Goal>> availableGoalsByCampaign;
        try {
            availableGoalsByCampaign = fetchGoalIdsByCampaign(
                    campaigns, container.getMetrikaClient(), operatorUid, clientId
            );
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
            vb.checkEach(metrikaReturnsResultWithErrors());
            return vb.getResult();
        }

        boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                .map(netAcl::isInternalIp)
                .orElse(false);
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        boolean allGoalsAreAvailable = requestFromInternalNetwork
                && !enabledFeatures.contains(UNIVERSAL_CAMPAIGNS_BETA_DISABLED.getName());

        vb.checkEachBy(campaign -> new CampaignWithMeaningfulGoalsValidator(
                availableGoalsByCampaign.get(campaign), container, allGoalsAreAvailable, enabledFeatures)
                .apply(campaign));

        return vb.getResult();
    }

    private Map<CampaignWithMeaningfulGoalsWithRequiredFields, Set<Goal>> fetchGoalIdsByCampaign(
            List<CampaignWithMeaningfulGoalsWithRequiredFields> campaigns,
            RequestBasedMetrikaClientAdapter metrikaData,
            Long operatorUid, ClientId clientId) {
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        Map<CampaignWithMeaningfulGoalsWithRequiredFields, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaign =
                StreamEx.of(campaigns)
                        .mapToEntry(campaign -> toCampaignTypeWithCounterIds(campaign,
                                new HashSet<>(nvl(campaign.getMetrikaCounters(), Set.of())), enabledFeatures))
                        .distinctKeys()
                        .toMap();

        Map<CampaignTypeWithCounterIds, Set<Goal>> goalsByCampaignTypeWithCounterId =
                campaignGoalsService.getAvailableGoalsForCampaignType(
                        operatorUid,
                        clientId,
                        Set.copyOf(campaignTypeWithCounterIdsByCampaign.values()),
                        metrikaData
                );

        return EntryStream.of(campaignTypeWithCounterIdsByCampaign)
                .mapValues(campaignTypeWithCounterIds -> getGoalIdsByCampaign(goalsByCampaignTypeWithCounterId,
                        campaignTypeWithCounterIds))
                .toMap();
    }

    private Set<Goal> getGoalIdsByCampaign(Map<CampaignTypeWithCounterIds, Set<Goal>> goalsByCampaign,
                                           CampaignTypeWithCounterIds campaign) {
        return goalsByCampaign.getOrDefault(campaign, Set.of());
    }
}
