package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrganization;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.organizations.validation.OrganizationConstraints.checkStatus;
import static ru.yandex.direct.core.entity.organizations.validation.OrganizationDefects.eitherPermalinkOrChainCanBeFilled;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithOrganizationAddValidationTypeSupport
        extends AbstractCampaignAddValidationTypeSupport<CampaignWithOrganization> {

    private final OrganizationService organizationService;

    @Autowired
    CampaignWithOrganizationAddValidationTypeSupport(OrganizationService organizationService) {
        this.organizationService = organizationService;
    }

    @Override
    public Class<CampaignWithOrganization> getTypeClass() {
        return CampaignWithOrganization.class;
    }

    @Override
    public ValidationResult<List<CampaignWithOrganization>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithOrganization>, Defect> vr) {

        Validator<CampaignWithOrganization, Defect> validator = campaign -> {
            ModelItemValidationBuilder<CampaignWithOrganization> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(CampaignWithOrganization.DEFAULT_PERMALINK_ID)
                    .check(validId(), When.notNull());
            vb.item(CampaignWithOrganization.DEFAULT_CHAIN_ID)
                    .check(validId(), When.notNull());
            vb.check(Constraint.fromPredicate(this::validatePermalinkChainUniqueness,
                    eitherPermalinkOrChainCanBeFilled()),
                    When.isValid());
            return vb.getResult();
        };

        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator)
                .getResult();
    }

    /**
     * Проверяет, что из двух полей {@link CampaignWithOrganization#getDefaultPermalinkId()} и
     * {@link CampaignWithOrganization#getDefaultChainId()} заполнено максимум одно.
     */
    private boolean validatePermalinkChainUniqueness(CampaignWithOrganization campaign) {
        return campaign.getDefaultPermalinkId() == null || campaign.getDefaultChainId() == null;
    }

    @Override
    public ValidationResult<List<CampaignWithOrganization>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithOrganization>, Defect> vr) {

        List<Long> permalinkIds = StreamEx.of(ValidationResult.getValidItems(vr))
                .map(CampaignWithOrganization::getDefaultPermalinkId)
                .toList();

        if (permalinkIds.isEmpty()) {
            return vr;
        }

        Map<Long, Organization> organizations =
                organizationService.getClientOrganizations(permalinkIds, container.getClientId());

        Validator<CampaignWithOrganization, Defect> validator = campaign -> {
            ModelItemValidationBuilder<CampaignWithOrganization> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(CampaignWithOrganization.DEFAULT_PERMALINK_ID)
                    .check(checkStatus(organizations, container.isCopy()), When.notNull());
            // todo: проверять, что сеть существует в Справочнике и подходит нам
            //  (подходит = она непустая, или у неё есть хотя бы одна неопубликованная организация, или как-то ещё)
            return vb.getResult();
        };

        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator, When.isValid())
                .getResult();
    }
}
