package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithPackageStrategyAddPreValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithPackageStrategyAddValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCampaignIds;
import ru.yandex.direct.core.entity.strategy.service.converter.CampaignToStrategyConverterService;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.common.db.PpcPropertyNames.MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.feature.FeatureName.PACKAGE_STRATEGIES_STAGE_TWO;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPackageStrategyAddValidationTypeSupport extends AbstractCampaignAddValidationTypeSupport<CampaignWithPackageStrategy> {

    private final FeatureService featureService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignToStrategyConverterService campaignToStrategyConverterService;

    @Autowired
    public CampaignWithPackageStrategyAddValidationTypeSupport(FeatureService featureService,
                                                               PpcPropertiesSupport ppcPropertiesSupport,
                                                               CampaignTypedRepository campaignTypedRepository,
                                                               CampaignToStrategyConverterService campaignToStrategyConverterService) {
        this.featureService = featureService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignToStrategyConverterService = campaignToStrategyConverterService;
    }

    @Override
    public Class<CampaignWithPackageStrategy> getTypeClass() {
        return CampaignWithPackageStrategy.class;
    }

    @Override
    public ValidationResult<List<CampaignWithPackageStrategy>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithPackageStrategy>, Defect> vr) {
        boolean isPackageStrategiesStageTwoEnabled = featureService.isEnabledForClientId(container.getClientId(),
                PACKAGE_STRATEGIES_STAGE_TWO);

        if (!isPackageStrategiesStageTwoEnabled) {
            return vr;
        }

        Map<Long, BaseStrategy> strategyById = StreamEx.of(vr.getValue())
                .map(CampaignWithPackageStrategy::getStrategyId)
                .distinct()
                .filter(CommonUtils::isValidId)
                .mapToEntry(container::getPackageStrategy)
                .nonNullValues()
                .toMap();

        Set<Long> availableStrategiesFromOperation = strategyById.keySet();

        var now = now();
        Map<CampaignWithPackageStrategy, BaseStrategy> strategyFromCampaignStrategyByCampaign =
                StreamEx.of(vr.getValue())
                        .filter(c -> c.getStrategyId() != null && availableStrategiesFromOperation.contains(c.getStrategyId())
                                && c.getStrategy() != null && c.getStrategy().getStrategyData() != null)
                        .mapToEntry(c -> campaignToStrategyConverterService
                                .toStrategyWithIdEqualToCampaignStrategyId(container.getClientId(), now, c))
                        .toMap();

        var validator = createPreValidator(strategyById, strategyFromCampaignStrategyByCampaign);
        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator)
                .getResult();
    }

    @Override
    public ValidationResult<List<CampaignWithPackageStrategy>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithPackageStrategy>, Defect> vr) {
        boolean isPackageStrategiesStageTwoEnabled = featureService.isEnabledForClientId(container.getClientId(),
                PACKAGE_STRATEGIES_STAGE_TWO);

        if (!isPackageStrategiesStageTwoEnabled) {
            return vr;
        }

        Map<Long, BaseStrategy> newCampaignStrategyById = StreamEx.of(vr.getValue())
                .map(CampaignWithPackageStrategy::getStrategyId)
                .distinct()
                .filter(strategyId -> strategyId != 0)
                .mapToEntry(container::getPackageStrategy)
                .toMap();

        Map<Long, Long> oneOfAlreadyLinkedToStrategyCampaignsIdByStrategyId = EntryStream.of(newCampaignStrategyById)
                .mapValues(s -> ((StrategyWithCampaignIds) s).getCids())
                .nonNullValues()
                .mapValues(cids -> cids.get(0))
                .toMap();

        Map<Long, BaseCampaign> linkedCampaignsById =
                campaignTypedRepository.getIdToModelTyped(
                        container.getShard(), oneOfAlreadyLinkedToStrategyCampaignsIdByStrategyId.values());

        Map<Long, CampaignType> linkedCampaignsTypeById = EntryStream.of(linkedCampaignsById)
                .mapValues(c -> ((CampaignWithCampaignType) c).getType())
                .toMap();

        Map<Long, CampaignType> linkedCampaignsTypeByStrategyId =
                EntryStream.of(oneOfAlreadyLinkedToStrategyCampaignsIdByStrategyId)
                        .mapValues(linkedCampaignsTypeById::get)
                        .nonNullValues()
                        .toMap();

        var validator = createValidator(vr.getValue(), newCampaignStrategyById, linkedCampaignsTypeByStrategyId);
        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator)
                .getResult();
    }

    private Validator<CampaignWithPackageStrategy, Defect> createPreValidator(
            Map<Long, BaseStrategy> strategyById,
            Map<CampaignWithPackageStrategy, BaseStrategy> strategyFromCampaignStrategyByCampaign) {
        return campaign -> {
            BaseStrategy strategyWithCampaignStrategyId = null;
            BaseStrategy strategyFromCampaignStrategyWithCampaignStrategyId = null;

            if (campaign.getStrategyId() != null) {
                strategyWithCampaignStrategyId = strategyById.get(campaign.getStrategyId());
                strategyFromCampaignStrategyWithCampaignStrategyId =
                        strategyFromCampaignStrategyByCampaign.get(campaign);
            }

            return new CampaignWithPackageStrategyAddPreValidator(strategyWithCampaignStrategyId,
                    strategyFromCampaignStrategyWithCampaignStrategyId).apply(campaign);
        };
    }

    private Validator<CampaignWithPackageStrategy, Defect> createValidator(List<CampaignWithPackageStrategy> campaigns,
                                                                           Map<Long, BaseStrategy> newCampaignStrategyById,
                                                                           Map<Long, CampaignType> linkedCampaignsTypeByStrategyId) {
        var maxNumberOfLinkedCids = ppcPropertiesSupport.get(MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY)
                .getOrDefault(DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY);
        return campaign ->
                new CampaignWithPackageStrategyAddValidator(
                        newCampaignStrategyById.get(campaign.getStrategyId()),
                        linkedCampaignsTypeByStrategyId.get(campaign.getStrategyId()), campaigns,
                        maxNumberOfLinkedCids).apply(campaign);
    }
}
