package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.service.validation.type.CampaignWithPricePackageValidationUtils;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithPricePackageValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageWithoutClients;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPricePackageAddValidationTypeSupport
        extends AbstractCampaignAddValidationTypeSupport<CampaignWithPricePackage> {

    private final PricePackageService pricePackageService;
    private final FeatureService featureService;

    @Autowired
    public CampaignWithPricePackageAddValidationTypeSupport(
            PricePackageService pricePackageService,
            FeatureService featureService
    ) {
        this.pricePackageService = pricePackageService;
        this.featureService = featureService;
    }

    @Override
    public Class<CampaignWithPricePackage> getTypeClass() {
        return CampaignWithPricePackage.class;
    }

    @Override
    public ValidationResult<List<CampaignWithPricePackage>, Defect> preValidate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithPricePackage>, Defect> vr) {
        Set<Long> pricePackageIds = CampaignWithPricePackageValidationUtils.extractPricePackageIds(vr.getValue());
        List<PricePackageWithoutClients> pricePackages = pricePackageService.getActivePricePackagesWithoutClients(
                container.getClientId(), pricePackageIds);
        Map<Long, PricePackageWithoutClients> pricePackagesMap = listToMap(pricePackages,
                PricePackageWithoutClients::getId);
        return new ListValidationBuilder<>(vr)
                .checkEachBy(campaign -> {
                    var vb = ModelItemValidationBuilder.of(campaign);

                    vb.item(CampaignWithPricePackage.FLIGHT_STATUS_APPROVE)
                            .check(isNull());
                    vb.item(CampaignWithPricePackage.FLIGHT_STATUS_CORRECT)
                            .check(isNull());
                    vb.item(CampaignWithPricePackage.FLIGHT_REASON_INCORRECT)
                            .check(isNull());

                    vb.item(CampaignWithPricePackage.STRATEGY)
                            .check(isNull());

                    return vb.getResult();
                })
                // важно перед сверкой с параметрами пакета проверить, что пакет доступен клиенту
                // иначе можно раскрыть параметры чужого пакета.
                .checkEachBy(campaign -> validatePackageExistsAndAvailable(pricePackagesMap, campaign))
                .getResult();
    }

    @Override
    public ValidationResult<List<CampaignWithPricePackage>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CampaignWithPricePackage>, Defect> vr) {
        Set<Long> pricePackageIds = CampaignWithPricePackageValidationUtils.extractPricePackageIds(vr.getValue());
        List<PricePackageWithoutClients> pricePackages = pricePackageService.getActivePricePackagesWithoutClients(
                container.getClientId(), pricePackageIds);
        Map<Long, PricePackageWithoutClients> pricePackagesMap = listToMap(pricePackages,
                PricePackageWithoutClients::getId);
        Set<String> availableFeatures = featureService.getEnabledForClientId(container.getClientId());
        var commonValidator = new CampaignWithPricePackageValidator<>(pricePackagesMap, availableFeatures);
        return new ListValidationBuilder<>(vr)
                .checkEachBy(commonValidator, When.isValid())
                .getResult();
    }

    private ValidationResult<CampaignWithPricePackage, Defect> validatePackageExistsAndAvailable(
            Map<Long, PricePackageWithoutClients> pricePackages, CampaignWithPricePackage campaign) {
        PricePackageWithoutClients pricePackage = ifNotNull(campaign.getPricePackageId(), pricePackages::get);

        ModelItemValidationBuilder<CampaignWithPricePackage> vb = ModelItemValidationBuilder.of(campaign);
        vb.item(CampaignWithPricePackage.PRICE_PACKAGE_ID)
                .check(notNull())
                .check(fromPredicate(pricePackageId -> pricePackage != null, objectNotFound()), When.isValid());

        return vb.getResult();
    }
}
