package ru.yandex.direct.core.entity.campaign.service.validation.type.add;

import java.time.LocalDate;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.service.type.add.AddServicedCampaignService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.AddServicedCampaignInfo;
import ru.yandex.direct.core.entity.campaign.service.validation.type.CommonCampaignBeanValidatorContext;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CommonCampaignBeanValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.time.LocalDate.now;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.requireServicingWillBeIgnored;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;

@Component
@ParametersAreNonnullByDefault
public class CommonCampaignAddValidationTypeSupport extends AbstractCampaignAddValidationTypeSupport<CommonCampaign> {
    private final NetAcl netAcl;
    private final AddServicedCampaignService addServicedCampaignService;

    @Autowired
    public CommonCampaignAddValidationTypeSupport(
            NetAcl netAcl,
            AddServicedCampaignService addServicedCampaignService) {
        this.netAcl = netAcl;
        this.addServicedCampaignService = addServicedCampaignService;
    }

    @Override
    public Class<CommonCampaign> getTypeClass() {
        return CommonCampaign.class;
    }

    @Override
    public ValidationResult<List<CommonCampaign>, Defect> preValidate(CampaignValidationContainer container,
                                                                      ValidationResult<List<CommonCampaign>, Defect> vr) {
        var vb = new ListValidationBuilder<>(vr);

        vb.checkEachBy(x -> new CommonCampaignSystemFieldsPreValidator().apply(x));

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<CommonCampaign>, Defect> validate(
            CampaignValidationContainer container,
            ValidationResult<List<CommonCampaign>, Defect> vr) {
        var vb = new ListValidationBuilder<>(vr);
        LocalDate now = now();

        List<CommonCampaign> campaigns = vr.getValue();
        List<AddServicedCampaignInfo> servicedInfos =
                addServicedCampaignService.getServicedInfoForNewCampaigns(container, campaigns);
        Map<CommonCampaign, AddServicedCampaignInfo> campaignToServiceInfoMap =
                EntryStream.zip(campaigns, servicedInfos)
                        .toCustomMap(IdentityHashMap::new);

        CommonCampaignBeanValidatorContext context = new CommonCampaignBeanValidatorContext(netAcl);
        vb
                .checkEachBy(x -> CommonCampaignIntervalValidator.build(now).apply(x),
                        When.isFalse(container.isCopy()))
                .checkEachBy(x -> CommonCampaignBeanValidator.build(context,
                        container.getOptions().isLimitFioTo255Chars(),
                        container.getOptions().isValidateFioForForbiddenChars(),
                        container.getOptions().isValidateCampaignNameForForbiddenChars()
                )
                        .apply(x))
                .weakCheckEach(requireServicingIsNotIgnored(campaignToServiceInfoMap, container));

        return vb.getResult();
    }

    private Constraint<CommonCampaign, Defect> requireServicingIsNotIgnored(
            Map<CommonCampaign, AddServicedCampaignInfo> campaignToServiceInfoMap,
            CampaignValidationContainer container) {
        return Constraint.fromPredicate(campaign -> {
            var notOk = container.getRequireServicing(campaign) == Boolean.TRUE
                    && campaignToServiceInfoMap.get(campaign).getIsServiced() == Boolean.FALSE;
            return !notOk;
        }, requireServicingWillBeIgnored());
    }

    private static class CommonCampaignIntervalValidator implements Validator<CommonCampaign, Defect> {
        private final LocalDate now;

        private CommonCampaignIntervalValidator(LocalDate now) {
            this.now = now;
        }

        public static CommonCampaignIntervalValidator build(LocalDate now) {
            return new CommonCampaignIntervalValidator(now);
        }

        @Override
        public ValidationResult<CommonCampaign, Defect> apply(CommonCampaign campaign) {
            ModelItemValidationBuilder<CommonCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(CommonCampaign.START_DATE)
                    .check(isNotBeforeThan(now), When.notNull());
            vb.item(CommonCampaign.END_DATE)
                    .check(isNotBeforeThan(now), When.notNull());
            return vb.getResult();
        }
    }

    private static class CommonCampaignSystemFieldsPreValidator implements Validator<CommonCampaign, Defect> {

        @Override
        public ValidationResult<CommonCampaign, Defect> apply(CommonCampaign campaign) {
            ModelItemValidationBuilder<CommonCampaign> vb = ModelItemValidationBuilder.of(campaign);
            vb.item(CommonCampaign.AGENCY_ID)
                    .check(isNull());
            vb.item(CommonCampaign.AGENCY_UID)
                    .check(isNull());
            vb.item(CommonCampaign.CLIENT_ID)
                    .check(isNull());
            vb.item(CommonCampaign.WALLET_ID)
                    .check(isNull());
            vb.item(CommonCampaign.IS_SERVICE_REQUESTED)
                    .check(isNull());
            return vb.getResult();
        }
    }

}
